<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // disable direct access
}

use function FortAwesome\fa;

if ( ! class_exists('Mega_Menu_Font_Awesome_6') ) :

/**
 *
 */
class Mega_Menu_Font_Awesome_6 {

	/**
	 * Constructor
	 *
	 * @since 2.3
	 */
	public function __construct() {
		add_filter( 'megamenu_load_scss_file_contents', array( $this, 'append_fontawesome_scss'), 10 );
		add_filter( 'megamenu_icon_tabs', array( $this, 'font_awesome_selector'), 99, 5 );
		add_action( 'megamenu_enqueue_public_scripts', array ( $this, 'enqueue_public_scripts'), 10 );
        add_action( "megamenu_admin_scripts", array( $this, 'enqueue_admin_styles') );
		add_action( "admin_print_scripts-nav-menus.php", array( $this, 'enqueue_admin_styles' ) );

		//$this->fa_icons();
	}

	
	/**
	 * Returns true when FontAwesome 6 Pro is installed/enabled
	 * 
	 * @since 2.3
	 */
	public function use_pro() {
		if ( defined( "MEGAMENU_PRO_USE_FONTAWESOME6_PRO" ) ) {
			return MEGAMENU_PRO_USE_FONTAWESOME6_PRO;
		}

		if ( function_exists( 'FortAwesome\fa' ) ) {
			$version = fa()->version();
			$version_is_v6 = is_string( $version ) && 1 === preg_match( '/^6\./', $version );

			if ( fa()->pro() && $version_is_v6 ) {
				return true;
			}
		}

		return false;
	}


    /**
     * Enqueue required CSS and JS for Mega Menu
     *
     */
    public function enqueue_admin_styles( $hook ) {

    	if ( $this->use_pro() ) {
    		return;
    	}

        wp_enqueue_style( 'megamenu-fontawesome6', plugins_url( 'css/all.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
    }

	/**
	 * Add the CSS required to display fontawesome icons to the main SCSS file
	 *
	 * @since 2.3
	 * @param string $scss
	 * @return string
	 */
	public function append_fontawesome_scss( $scss ) {
		$path = trailingslashit( plugin_dir_path( __FILE__ ) ) . 'scss/fontawesome.scss';
		$contents = file_get_contents( $path );
 		return $scss . $contents;
	}


	/**
	 * Enqueue fontawesome CSS
	 *
	 * @since 2.3
	 */
	public function enqueue_public_scripts() {
		$settings = get_option("megamenu_settings");

        if ( is_array( $settings ) && isset( $settings['enqueue_fa_6'] ) && $settings['enqueue_fa_6'] == 'disabled' ) {
        	return;
        }

        wp_enqueue_style( 'megamenu-fontawesome6', plugins_url( 'css/all.min.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
	}


	/**
	 * Generate HTML for the font awesome selector
	 *
	 * @since 2.3
	 * @param array $tabs
	 * @param int $menu_item_id
	 * @param int $menu_id
	 * @param int $menu_item_depth
	 * @param array $menu_item_meta
	 * @return array
	 */
	public function font_awesome_selector( $tabs, $menu_item_id, $menu_id, $menu_item_depth, $menu_item_meta ) {
		$is_pro = $this->use_pro() ? "pro" : "";
		$settings = get_option("megamenu_settings");
        $css_version = get_option("megamenu_pro_css_version");
        $html = "";

        if ( is_array( $settings ) && isset( $settings['enqueue_fa_6'] ) && $settings['enqueue_fa_6'] == 'disabled' && !$this->use_pro() ) {
        	$html .= "<div class='notice notice-warning'>" . __("Font Awesome 6 has been dequeued under Mega Menu > General Settings. You will need to ensure that Font Awesome 6 is enqueued on your site using an alternative method.", "megamenu-pro") . "</div>";
        }

		if ( $css_version && version_compare( $css_version, '2.3', '<' ) ) {
			$link    = "<a href='" . esc_attr( admin_url( 'admin.php?page=maxmegamenu_tools' ) ) . "'>" . __( 'Mega Menu' ) . ' > ' . __( 'Tools' ) . '</a>';
			$html .= "<div class='notice notice-warning'><p>";
			$html .= sprintf( __( 'Your menu CSS needs to be updated to work with Font Awesome 6. Please go to %s and Clear the CSS Cache (you will only need to do this once).', 'megamenu' ), $link );
			$html .= '</p></div>';
		}

        foreach ( $this->icons() as $code => $class ) {

            $bits = explode( "-", $code );
            $code = "&#x" . $bits[1] . "";
            $type = $bits[0];

            $style_bits = explode( " ", $class);
            $style = $style_bits[0];

            $class = $class . " fa6";

            $html .= "<div class='{$style} {$is_pro}'>";
            $html .= "    <input class='radio' id='{$class}' type='radio' rel='{$code}' name='settings[icon]' value='{$class}' " . checked( $menu_item_meta['icon'], $class, false ) . " />";
            $html .= "    <label rel='{$code}' for='{$class}' title='{$class}'></label>";
            $html .= "</div>";

        }

        $icon_prefix = "";

        if ( isset( $menu_item_meta['icon'] ) ) {
        	$icon_prefix = substr( $menu_item_meta['icon'], 0, 3 );
        }

        $title = __("Font Awesome 6", "megamenu-pro");

        if ( $this->use_pro() ) {
        	$title .= " Pro";

        	if ( function_exists( 'FortAwesome\fa' ) ) {
        		$title .= " (Kit)";
        	}
        }

        $insert_position = 3;

        if ( ! Mega_Menu_Pro::font_awesome_4_is_enabled() ) {
        	unset($tabs['fontawesome']);
        	$insert_position = 1;
        }

		$insert = array(
			'fontawesome6' => array(
				'title' => $title,
				'active' => isset( $menu_item_meta['icon'] ) && str_contains($menu_item_meta['icon'], 'fa6' ),
				'content' => $html
			)
		);

		$this->array_splice_assoc( $tabs, $insert_position, 0, $insert );
		return $tabs;
	}


	public function array_splice_assoc(&$input, $offset, $length, $replacement = array()) {
	    $replacement = (array) $replacement;
	    $key_indices = array_flip(array_keys($input));
	    if (isset($input[$offset]) && is_string($offset)) {
	            $offset = $key_indices[$offset];
	    }
	    if (isset($input[$length]) && is_string($length)) {
	            $length = $key_indices[$length] - $offset;
	    }

	    $input = array_slice($input, 0, $offset, TRUE)
	            + $replacement
	            + array_slice($input, $offset + $length, NULL, TRUE); 
	}

	/**
	 * Return an array of Font Awesome icons
	 *
	 * @since 2.3
	 * @return array $icons
	 */
	public function icons() {

		$icons = array(
			'fa-e080' => 'fab fa-42-group',
			'fa-f26e' => 'fab fa-500px',
			'fa-41' => 'fas fa-a',
			'fa-f368' => 'fab fa-accessible-icon',
			'fa-f369' => 'fab fa-accusoft',
			'fa-f2b9' => 'fas fa-address-book',
			'fa-f2b9' => 'far fa-address-book',
			'fa-f2bb' => 'fas fa-address-card',
			'fa-f2bb' => 'far fa-address-card',
			'fa-f170' => 'fab fa-adn',
			'fa-f36a' => 'fab fa-adversal',
			'fa-f36b' => 'fab fa-affiliatetheme',
			'fa-f834' => 'fab fa-airbnb',
			'fa-f36c' => 'fab fa-algolia',
			'fa-f037' => 'fas fa-align-center',
			'fa-f039' => 'fas fa-align-justify',
			'fa-f036' => 'fas fa-align-left',
			'fa-f038' => 'fas fa-align-right',
			'fa-f642' => 'fab fa-alipay',
			'fa-f270' => 'fab fa-amazon',
			'fa-f42c' => 'fab fa-amazon-pay',
			'fa-f36d' => 'fab fa-amilia',
			'fa-f13d' => 'fas fa-anchor',
			'fa-e4aa' => 'fas fa-anchor-circle-check',
			'fa-e4ab' => 'fas fa-anchor-circle-exclamation',
			'fa-e4ac' => 'fas fa-anchor-circle-xmark',
			'fa-e4ad' => 'fas fa-anchor-lock',
			'fa-f17b' => 'fab fa-android',
			'fa-f209' => 'fab fa-angellist',
			'fa-f107' => 'fas fa-angle-down',
			'fa-f104' => 'fas fa-angle-left',
			'fa-f105' => 'fas fa-angle-right',
			'fa-f106' => 'fas fa-angle-up',
			'fa-f103' => 'fas fa-angles-down',
			'fa-f100' => 'fas fa-angles-left',
			'fa-f101' => 'fas fa-angles-right',
			'fa-f102' => 'fas fa-angles-up',
			'fa-f36e' => 'fab fa-angrycreative',
			'fa-f420' => 'fab fa-angular',
			'fa-f644' => 'fas fa-ankh',
			'fa-f36f' => 'fab fa-app-store',
			'fa-f370' => 'fab fa-app-store-ios',
			'fa-f371' => 'fab fa-apper',
			'fa-f179' => 'fab fa-apple',
			'fa-f415' => 'fab fa-apple-pay',
			'fa-f5d1' => 'fas fa-apple-whole',
			'fa-f557' => 'fas fa-archway',
			'fa-f063' => 'fas fa-arrow-down',
			'fa-f162' => 'fas fa-arrow-down-1-9',
			'fa-f886' => 'fas fa-arrow-down-9-1',
			'fa-f15d' => 'fas fa-arrow-down-a-z',
			'fa-f175' => 'fas fa-arrow-down-long',
			'fa-f884' => 'fas fa-arrow-down-short-wide',
			'fa-e4af' => 'fas fa-arrow-down-up-across-line',
			'fa-e4b0' => 'fas fa-arrow-down-up-lock',
			'fa-f160' => 'fas fa-arrow-down-wide-short',
			'fa-f881' => 'fas fa-arrow-down-z-a',
			'fa-f060' => 'fas fa-arrow-left',
			'fa-f177' => 'fas fa-arrow-left-long',
			'fa-f245' => 'fas fa-arrow-pointer',
			'fa-f061' => 'fas fa-arrow-right',
			'fa-f0ec' => 'fas fa-arrow-right-arrow-left',
			'fa-f08b' => 'fas fa-arrow-right-from-bracket',
			'fa-f178' => 'fas fa-arrow-right-long',
			'fa-f090' => 'fas fa-arrow-right-to-bracket',
			'fa-e4b3' => 'fas fa-arrow-right-to-city',
			'fa-f0e2' => 'fas fa-arrow-rotate-left',
			'fa-f01e' => 'fas fa-arrow-rotate-right',
			'fa-e097' => 'fas fa-arrow-trend-down',
			'fa-e098' => 'fas fa-arrow-trend-up',
			'fa-f149' => 'fas fa-arrow-turn-down',
			'fa-f148' => 'fas fa-arrow-turn-up',
			'fa-f062' => 'fas fa-arrow-up',
			'fa-f163' => 'fas fa-arrow-up-1-9',
			'fa-f887' => 'fas fa-arrow-up-9-1',
			'fa-f15e' => 'fas fa-arrow-up-a-z',
			'fa-e09a' => 'fas fa-arrow-up-from-bracket',
			'fa-e4b5' => 'fas fa-arrow-up-from-ground-water',
			'fa-e4b6' => 'fas fa-arrow-up-from-water-pump',
			'fa-f176' => 'fas fa-arrow-up-long',
			'fa-e4b7' => 'fas fa-arrow-up-right-dots',
			'fa-f08e' => 'fas fa-arrow-up-right-from-square',
			'fa-f885' => 'fas fa-arrow-up-short-wide',
			'fa-f161' => 'fas fa-arrow-up-wide-short',
			'fa-f882' => 'fas fa-arrow-up-z-a',
			'fa-e4b8' => 'fas fa-arrows-down-to-line',
			'fa-e4b9' => 'fas fa-arrows-down-to-people',
			'fa-f07e' => 'fas fa-arrows-left-right',
			'fa-e4ba' => 'fas fa-arrows-left-right-to-line',
			'fa-f021' => 'fas fa-arrows-rotate',
			'fa-e4bb' => 'fas fa-arrows-spin',
			'fa-e4bc' => 'fas fa-arrows-split-up-and-left',
			'fa-e4bd' => 'fas fa-arrows-to-circle',
			'fa-e4be' => 'fas fa-arrows-to-dot',
			'fa-e4bf' => 'fas fa-arrows-to-eye',
			'fa-e4c0' => 'fas fa-arrows-turn-right',
			'fa-e4c1' => 'fas fa-arrows-turn-to-dots',
			'fa-f07d' => 'fas fa-arrows-up-down',
			'fa-f047' => 'fas fa-arrows-up-down-left-right',
			'fa-e4c2' => 'fas fa-arrows-up-to-line',
			'fa-f77a' => 'fab fa-artstation',
			'fa-2a' => 'fas fa-asterisk',
			'fa-f372' => 'fab fa-asymmetrik',
			'fa-40' => 'fas fa-at',
			'fa-f77b' => 'fab fa-atlassian',
			'fa-f5d2' => 'fas fa-atom',
			'fa-f373' => 'fab fa-audible',
			'fa-f29e' => 'fas fa-audio-description',
			'fa-e0a9' => 'fas fa-austral-sign',
			'fa-f41c' => 'fab fa-autoprefixer',
			'fa-f374' => 'fab fa-avianex',
			'fa-f421' => 'fab fa-aviato',
			'fa-f559' => 'fas fa-award',
			'fa-f375' => 'fab fa-aws',
			'fa-42' => 'fas fa-b',
			'fa-f77c' => 'fas fa-baby',
			'fa-f77d' => 'fas fa-baby-carriage',
			'fa-f04a' => 'fas fa-backward',
			'fa-f049' => 'fas fa-backward-fast',
			'fa-f048' => 'fas fa-backward-step',
			'fa-f7e5' => 'fas fa-bacon',
			'fa-e059' => 'fas fa-bacteria',
			'fa-e05a' => 'fas fa-bacterium',
			'fa-f290' => 'fas fa-bag-shopping',
			'fa-f666' => 'fas fa-bahai',
			'fa-e0ac' => 'fas fa-baht-sign',
			'fa-f05e' => 'fas fa-ban',
			'fa-f54d' => 'fas fa-ban-smoking',
			'fa-f462' => 'fas fa-bandage',
			'fa-f2d5' => 'fab fa-bandcamp',
			'fa-e2e6' => 'fas fa-bangladeshi-taka-sign',
			'fa-f02a' => 'fas fa-barcode',
			'fa-f0c9' => 'fas fa-bars',
			'fa-f828' => 'fas fa-bars-progress',
			'fa-f550' => 'fas fa-bars-staggered',
			'fa-f433' => 'fas fa-baseball',
			'fa-f432' => 'fas fa-baseball-bat-ball',
			'fa-f291' => 'fas fa-basket-shopping',
			'fa-f434' => 'fas fa-basketball',
			'fa-f2cd' => 'fas fa-bath',
			'fa-f244' => 'fas fa-battery-empty',
			'fa-f240' => 'fas fa-battery-full',
			'fa-f242' => 'fas fa-battery-half',
			'fa-f243' => 'fas fa-battery-quarter',
			'fa-f241' => 'fas fa-battery-three-quarters',
			'fa-f835' => 'fab fa-battle-net',
			'fa-f236' => 'fas fa-bed',
			'fa-f487' => 'fas fa-bed-pulse',
			'fa-f0fc' => 'fas fa-beer-mug-empty',
			'fa-f1b4' => 'fab fa-behance',
			'fa-f0f3' => 'fas fa-bell',
			'fa-f0f3' => 'far fa-bell',
			'fa-f562' => 'fas fa-bell-concierge',
			'fa-f1f6' => 'fas fa-bell-slash',
			'fa-f1f6' => 'far fa-bell-slash',
			'fa-f55b' => 'fas fa-bezier-curve',
			'fa-f206' => 'fas fa-bicycle',
			'fa-e3d9' => 'fab fa-bilibili',
			'fa-f378' => 'fab fa-bimobject',
			'fa-f1e5' => 'fas fa-binoculars',
			'fa-f780' => 'fas fa-biohazard',
			'fa-f171' => 'fab fa-bitbucket',
			'fa-f379' => 'fab fa-bitcoin',
			'fa-e0b4' => 'fas fa-bitcoin-sign',
			'fa-f37a' => 'fab fa-bity',
			'fa-f27e' => 'fab fa-black-tie',
			'fa-f37b' => 'fab fa-blackberry',
			'fa-f517' => 'fas fa-blender',
			'fa-f6b6' => 'fas fa-blender-phone',
			'fa-f781' => 'fas fa-blog',
			'fa-f37c' => 'fab fa-blogger',
			'fa-f37d' => 'fab fa-blogger-b',
			'fa-f293' => 'fab fa-bluetooth',
			'fa-f294' => 'fab fa-bluetooth-b',
			'fa-f032' => 'fas fa-bold',
			'fa-f0e7' => 'fas fa-bolt',
			'fa-e0b7' => 'fas fa-bolt-lightning',
			'fa-f1e2' => 'fas fa-bomb',
			'fa-f5d7' => 'fas fa-bone',
			'fa-f55c' => 'fas fa-bong',
			'fa-f02d' => 'fas fa-book',
			'fa-f558' => 'fas fa-book-atlas',
			'fa-f647' => 'fas fa-book-bible',
			'fa-e0bb' => 'fas fa-book-bookmark',
			'fa-f66a' => 'fas fa-book-journal-whills',
			'fa-f7e6' => 'fas fa-book-medical',
			'fa-f518' => 'fas fa-book-open',
			'fa-f5da' => 'fas fa-book-open-reader',
			'fa-f687' => 'fas fa-book-quran',
			'fa-f6b7' => 'fas fa-book-skull',
			'fa-f827' => 'fas fa-book-tanakh',
			'fa-f02e' => 'fas fa-bookmark',
			'fa-f02e' => 'far fa-bookmark',
			'fa-f836' => 'fab fa-bootstrap',
			'fa-f84c' => 'fas fa-border-all',
			'fa-f850' => 'fas fa-border-none',
			'fa-f853' => 'fas fa-border-top-left',
			'fa-e4c3' => 'fas fa-bore-hole',
			'fa-e340' => 'fab fa-bots',
			'fa-e4c4' => 'fas fa-bottle-droplet',
			'fa-e4c5' => 'fas fa-bottle-water',
			'fa-e4c6' => 'fas fa-bowl-food',
			'fa-e2eb' => 'fas fa-bowl-rice',
			'fa-f436' => 'fas fa-bowling-ball',
			'fa-f466' => 'fas fa-box',
			'fa-f187' => 'fas fa-box-archive',
			'fa-f49e' => 'fas fa-box-open',
			'fa-e05b' => 'fas fa-box-tissue',
			'fa-e4c7' => 'fas fa-boxes-packing',
			'fa-f468' => 'fas fa-boxes-stacked',
			'fa-f2a1' => 'fas fa-braille',
			'fa-f5dc' => 'fas fa-brain',
			'fa-e46c' => 'fas fa-brazilian-real-sign',
			'fa-f7ec' => 'fas fa-bread-slice',
			'fa-e4c8' => 'fas fa-bridge',
			'fa-e4c9' => 'fas fa-bridge-circle-check',
			'fa-e4ca' => 'fas fa-bridge-circle-exclamation',
			'fa-e4cb' => 'fas fa-bridge-circle-xmark',
			'fa-e4cc' => 'fas fa-bridge-lock',
			'fa-e4ce' => 'fas fa-bridge-water',
			'fa-f0b1' => 'fas fa-briefcase',
			'fa-f469' => 'fas fa-briefcase-medical',
			'fa-f51a' => 'fas fa-broom',
			'fa-f458' => 'fas fa-broom-ball',
			'fa-f55d' => 'fas fa-brush',
			'fa-f15a' => 'fab fa-btc',
			'fa-e4cf' => 'fas fa-bucket',
			'fa-f837' => 'fab fa-buffer',
			'fa-f188' => 'fas fa-bug',
			'fa-e490' => 'fas fa-bug-slash',
			'fa-e4d0' => 'fas fa-bugs',
			'fa-f1ad' => 'fas fa-building',
			'fa-f1ad' => 'far fa-building',
			'fa-e4d1' => 'fas fa-building-circle-arrow-right',
			'fa-e4d2' => 'fas fa-building-circle-check',
			'fa-e4d3' => 'fas fa-building-circle-exclamation',
			'fa-e4d4' => 'fas fa-building-circle-xmark',
			'fa-f19c' => 'fas fa-building-columns',
			'fa-e4d5' => 'fas fa-building-flag',
			'fa-e4d6' => 'fas fa-building-lock',
			'fa-e4d7' => 'fas fa-building-ngo',
			'fa-e4d8' => 'fas fa-building-shield',
			'fa-e4d9' => 'fas fa-building-un',
			'fa-e4da' => 'fas fa-building-user',
			'fa-e4db' => 'fas fa-building-wheat',
			'fa-f0a1' => 'fas fa-bullhorn',
			'fa-f140' => 'fas fa-bullseye',
			'fa-f805' => 'fas fa-burger',
			'fa-f37f' => 'fab fa-buromobelexperte',
			'fa-e4dc' => 'fas fa-burst',
			'fa-f207' => 'fas fa-bus',
			'fa-f55e' => 'fas fa-bus-simple',
			'fa-f64a' => 'fas fa-business-time',
			'fa-f8a6' => 'fab fa-buy-n-large',
			'fa-f20d' => 'fab fa-buysellads',
			'fa-43' => 'fas fa-c',
			'fa-f7da' => 'fas fa-cable-car',
			'fa-f1fd' => 'fas fa-cake-candles',
			'fa-f1ec' => 'fas fa-calculator',
			'fa-f133' => 'fas fa-calendar',
			'fa-f133' => 'far fa-calendar',
			'fa-f274' => 'fas fa-calendar-check',
			'fa-f274' => 'far fa-calendar-check',
			'fa-f783' => 'fas fa-calendar-day',
			'fa-f073' => 'fas fa-calendar-days',
			'fa-f073' => 'far fa-calendar-days',
			'fa-f272' => 'fas fa-calendar-minus',
			'fa-f272' => 'far fa-calendar-minus',
			'fa-f271' => 'fas fa-calendar-plus',
			'fa-f271' => 'far fa-calendar-plus',
			'fa-f784' => 'fas fa-calendar-week',
			'fa-f273' => 'fas fa-calendar-xmark',
			'fa-f273' => 'far fa-calendar-xmark',
			'fa-f030' => 'fas fa-camera',
			'fa-f083' => 'fas fa-camera-retro',
			'fa-e0d8' => 'fas fa-camera-rotate',
			'fa-f6bb' => 'fas fa-campground',
			'fa-f785' => 'fab fa-canadian-maple-leaf',
			'fa-f786' => 'fas fa-candy-cane',
			'fa-f55f' => 'fas fa-cannabis',
			'fa-f46b' => 'fas fa-capsules',
			'fa-f1b9' => 'fas fa-car',
			'fa-f5df' => 'fas fa-car-battery',
			'fa-f5e1' => 'fas fa-car-burst',
			'fa-e4dd' => 'fas fa-car-on',
			'fa-f5de' => 'fas fa-car-rear',
			'fa-f5e4' => 'fas fa-car-side',
			'fa-e4de' => 'fas fa-car-tunnel',
			'fa-f8ff' => 'fas fa-caravan',
			'fa-f0d7' => 'fas fa-caret-down',
			'fa-f0d9' => 'fas fa-caret-left',
			'fa-f0da' => 'fas fa-caret-right',
			'fa-f0d8' => 'fas fa-caret-up',
			'fa-f787' => 'fas fa-carrot',
			'fa-f218' => 'fas fa-cart-arrow-down',
			'fa-f474' => 'fas fa-cart-flatbed',
			'fa-f59d' => 'fas fa-cart-flatbed-suitcase',
			'fa-f217' => 'fas fa-cart-plus',
			'fa-f07a' => 'fas fa-cart-shopping',
			'fa-f788' => 'fas fa-cash-register',
			'fa-f6be' => 'fas fa-cat',
			'fa-f42d' => 'fab fa-cc-amazon-pay',
			'fa-f1f3' => 'fab fa-cc-amex',
			'fa-f416' => 'fab fa-cc-apple-pay',
			'fa-f24c' => 'fab fa-cc-diners-club',
			'fa-f1f2' => 'fab fa-cc-discover',
			'fa-f24b' => 'fab fa-cc-jcb',
			'fa-f1f1' => 'fab fa-cc-mastercard',
			'fa-f1f4' => 'fab fa-cc-paypal',
			'fa-f1f5' => 'fab fa-cc-stripe',
			'fa-f1f0' => 'fab fa-cc-visa',
			'fa-e0df' => 'fas fa-cedi-sign',
			'fa-e3f5' => 'fas fa-cent-sign',
			'fa-f380' => 'fab fa-centercode',
			'fa-f789' => 'fab fa-centos',
			'fa-f0a3' => 'fas fa-certificate',
			'fa-f6c0' => 'fas fa-chair',
			'fa-f51b' => 'fas fa-chalkboard',
			'fa-f51c' => 'fas fa-chalkboard-user',
			'fa-f79f' => 'fas fa-champagne-glasses',
			'fa-f5e7' => 'fas fa-charging-station',
			'fa-f1fe' => 'fas fa-chart-area',
			'fa-f080' => 'fas fa-chart-bar',
			'fa-f080' => 'far fa-chart-bar',
			'fa-e0e3' => 'fas fa-chart-column',
			'fa-e0e4' => 'fas fa-chart-gantt',
			'fa-f201' => 'fas fa-chart-line',
			'fa-f200' => 'fas fa-chart-pie',
			'fa-e473' => 'fas fa-chart-simple',
			'fa-f00c' => 'fas fa-check',
			'fa-f560' => 'fas fa-check-double',
			'fa-f772' => 'fas fa-check-to-slot',
			'fa-f7ef' => 'fas fa-cheese',
			'fa-f439' => 'fas fa-chess',
			'fa-f43a' => 'fas fa-chess-bishop',
			'fa-f43a' => 'far fa-chess-bishop',
			'fa-f43c' => 'fas fa-chess-board',
			'fa-f43f' => 'fas fa-chess-king',
			'fa-f43f' => 'far fa-chess-king',
			'fa-f441' => 'fas fa-chess-knight',
			'fa-f441' => 'far fa-chess-knight',
			'fa-f443' => 'fas fa-chess-pawn',
			'fa-f443' => 'far fa-chess-pawn',
			'fa-f445' => 'fas fa-chess-queen',
			'fa-f445' => 'far fa-chess-queen',
			'fa-f447' => 'fas fa-chess-rook',
			'fa-f447' => 'far fa-chess-rook',
			'fa-f078' => 'fas fa-chevron-down',
			'fa-f053' => 'fas fa-chevron-left',
			'fa-f054' => 'fas fa-chevron-right',
			'fa-f077' => 'fas fa-chevron-up',
			'fa-f1ae' => 'fas fa-child',
			'fa-e4e0' => 'fas fa-child-combatant',
			'fa-e59c' => 'fas fa-child-dress',
			'fa-e59d' => 'fas fa-child-reaching',
			'fa-e4e1' => 'fas fa-children',
			'fa-f268' => 'fab fa-chrome',
			'fa-f838' => 'fab fa-chromecast',
			'fa-f51d' => 'fas fa-church',
			'fa-f111' => 'fas fa-circle',
			'fa-f111' => 'far fa-circle',
			'fa-f0ab' => 'fas fa-circle-arrow-down',
			'fa-f0a8' => 'fas fa-circle-arrow-left',
			'fa-f0a9' => 'fas fa-circle-arrow-right',
			'fa-f0aa' => 'fas fa-circle-arrow-up',
			'fa-f058' => 'fas fa-circle-check',
			'fa-f058' => 'far fa-circle-check',
			'fa-f13a' => 'fas fa-circle-chevron-down',
			'fa-f137' => 'fas fa-circle-chevron-left',
			'fa-f138' => 'fas fa-circle-chevron-right',
			'fa-f139' => 'fas fa-circle-chevron-up',
			'fa-f4b9' => 'fas fa-circle-dollar-to-slot',
			'fa-f192' => 'fas fa-circle-dot',
			'fa-f192' => 'far fa-circle-dot',
			'fa-f358' => 'fas fa-circle-down',
			'fa-f358' => 'far fa-circle-down',
			'fa-f06a' => 'fas fa-circle-exclamation',
			'fa-f47e' => 'fas fa-circle-h',
			'fa-f042' => 'fas fa-circle-half-stroke',
			'fa-f05a' => 'fas fa-circle-info',
			'fa-f359' => 'fas fa-circle-left',
			'fa-f359' => 'far fa-circle-left',
			'fa-f056' => 'fas fa-circle-minus',
			'fa-e4e2' => 'fas fa-circle-nodes',
			'fa-f1ce' => 'fas fa-circle-notch',
			'fa-f28b' => 'fas fa-circle-pause',
			'fa-f28b' => 'far fa-circle-pause',
			'fa-f144' => 'fas fa-circle-play',
			'fa-f144' => 'far fa-circle-play',
			'fa-f055' => 'fas fa-circle-plus',
			'fa-f059' => 'fas fa-circle-question',
			'fa-f059' => 'far fa-circle-question',
			'fa-f7ba' => 'fas fa-circle-radiation',
			'fa-f35a' => 'fas fa-circle-right',
			'fa-f35a' => 'far fa-circle-right',
			'fa-f28d' => 'fas fa-circle-stop',
			'fa-f28d' => 'far fa-circle-stop',
			'fa-f35b' => 'fas fa-circle-up',
			'fa-f35b' => 'far fa-circle-up',
			'fa-f2bd' => 'fas fa-circle-user',
			'fa-f2bd' => 'far fa-circle-user',
			'fa-f057' => 'fas fa-circle-xmark',
			'fa-f057' => 'far fa-circle-xmark',
			'fa-f64f' => 'fas fa-city',
			'fa-e131' => 'fas fa-clapperboard',
			'fa-f328' => 'fas fa-clipboard',
			'fa-f328' => 'far fa-clipboard',
			'fa-f46c' => 'fas fa-clipboard-check',
			'fa-f46d' => 'fas fa-clipboard-list',
			'fa-e4e3' => 'fas fa-clipboard-question',
			'fa-f7f3' => 'fas fa-clipboard-user',
			'fa-f017' => 'fas fa-clock',
			'fa-f017' => 'far fa-clock',
			'fa-f1da' => 'fas fa-clock-rotate-left',
			'fa-f24d' => 'fas fa-clone',
			'fa-f24d' => 'far fa-clone',
			'fa-f20a' => 'fas fa-closed-captioning',
			'fa-f20a' => 'far fa-closed-captioning',
			'fa-f0c2' => 'fas fa-cloud',
			'fa-f0ed' => 'fas fa-cloud-arrow-down',
			'fa-f0ee' => 'fas fa-cloud-arrow-up',
			'fa-f76c' => 'fas fa-cloud-bolt',
			'fa-f73b' => 'fas fa-cloud-meatball',
			'fa-f6c3' => 'fas fa-cloud-moon',
			'fa-f73c' => 'fas fa-cloud-moon-rain',
			'fa-f73d' => 'fas fa-cloud-rain',
			'fa-f740' => 'fas fa-cloud-showers-heavy',
			'fa-e4e4' => 'fas fa-cloud-showers-water',
			'fa-f6c4' => 'fas fa-cloud-sun',
			'fa-f743' => 'fas fa-cloud-sun-rain',
			'fa-e07d' => 'fab fa-cloudflare',
			'fa-f383' => 'fab fa-cloudscale',
			'fa-f384' => 'fab fa-cloudsmith',
			'fa-f385' => 'fab fa-cloudversify',
			'fa-e139' => 'fas fa-clover',
			'fa-e360' => 'fab fa-cmplid',
			'fa-f121' => 'fas fa-code',
			'fa-f126' => 'fas fa-code-branch',
			'fa-f386' => 'fas fa-code-commit',
			'fa-e13a' => 'fas fa-code-compare',
			'fa-e13b' => 'fas fa-code-fork',
			'fa-f387' => 'fas fa-code-merge',
			'fa-e13c' => 'fas fa-code-pull-request',
			'fa-f1cb' => 'fab fa-codepen',
			'fa-f284' => 'fab fa-codiepie',
			'fa-f51e' => 'fas fa-coins',
			'fa-e140' => 'fas fa-colon-sign',
			'fa-f075' => 'fas fa-comment',
			'fa-f075' => 'far fa-comment',
			'fa-f651' => 'fas fa-comment-dollar',
			'fa-f4ad' => 'fas fa-comment-dots',
			'fa-f4ad' => 'far fa-comment-dots',
			'fa-f7f5' => 'fas fa-comment-medical',
			'fa-f4b3' => 'fas fa-comment-slash',
			'fa-f7cd' => 'fas fa-comment-sms',
			'fa-f086' => 'fas fa-comments',
			'fa-f086' => 'far fa-comments',
			'fa-f653' => 'fas fa-comments-dollar',
			'fa-f51f' => 'fas fa-compact-disc',
			'fa-f14e' => 'fas fa-compass',
			'fa-f14e' => 'far fa-compass',
			'fa-f568' => 'fas fa-compass-drafting',
			'fa-f066' => 'fas fa-compress',
			'fa-e4e5' => 'fas fa-computer',
			'fa-f8cc' => 'fas fa-computer-mouse',
			'fa-f78d' => 'fab fa-confluence',
			'fa-f20e' => 'fab fa-connectdevelop',
			'fa-f26d' => 'fab fa-contao',
			'fa-f563' => 'fas fa-cookie',
			'fa-f564' => 'fas fa-cookie-bite',
			'fa-f0c5' => 'fas fa-copy',
			'fa-f0c5' => 'far fa-copy',
			'fa-f1f9' => 'fas fa-copyright',
			'fa-f1f9' => 'far fa-copyright',
			'fa-f89e' => 'fab fa-cotton-bureau',
			'fa-f4b8' => 'fas fa-couch',
			'fa-f6c8' => 'fas fa-cow',
			'fa-f388' => 'fab fa-cpanel',
			'fa-f25e' => 'fab fa-creative-commons',
			'fa-f4e7' => 'fab fa-creative-commons-by',
			'fa-f4e8' => 'fab fa-creative-commons-nc',
			'fa-f4e9' => 'fab fa-creative-commons-nc-eu',
			'fa-f4ea' => 'fab fa-creative-commons-nc-jp',
			'fa-f4eb' => 'fab fa-creative-commons-nd',
			'fa-f4ec' => 'fab fa-creative-commons-pd',
			'fa-f4ed' => 'fab fa-creative-commons-pd-alt',
			'fa-f4ee' => 'fab fa-creative-commons-remix',
			'fa-f4ef' => 'fab fa-creative-commons-sa',
			'fa-f4f0' => 'fab fa-creative-commons-sampling',
			'fa-f4f1' => 'fab fa-creative-commons-sampling-plus',
			'fa-f4f2' => 'fab fa-creative-commons-share',
			'fa-f4f3' => 'fab fa-creative-commons-zero',
			'fa-f09d' => 'fas fa-credit-card',
			'fa-f09d' => 'far fa-credit-card',
			'fa-f6c9' => 'fab fa-critical-role',
			'fa-f125' => 'fas fa-crop',
			'fa-f565' => 'fas fa-crop-simple',
			'fa-f654' => 'fas fa-cross',
			'fa-f05b' => 'fas fa-crosshairs',
			'fa-f520' => 'fas fa-crow',
			'fa-f521' => 'fas fa-crown',
			'fa-f7f7' => 'fas fa-crutch',
			'fa-e152' => 'fas fa-cruzeiro-sign',
			'fa-f13c' => 'fab fa-css3',
			'fa-f38b' => 'fab fa-css3-alt',
			'fa-f1b2' => 'fas fa-cube',
			'fa-f1b3' => 'fas fa-cubes',
			'fa-e4e6' => 'fas fa-cubes-stacked',
			'fa-f38c' => 'fab fa-cuttlefish',
			'fa-44' => 'fas fa-d',
			'fa-f38d' => 'fab fa-d-and-d',
			'fa-f6ca' => 'fab fa-d-and-d-beyond',
			'fa-e052' => 'fab fa-dailymotion',
			'fa-f210' => 'fab fa-dashcube',
			'fa-f1c0' => 'fas fa-database',
			'fa-e60b' => 'fab fa-debian',
			'fa-e077' => 'fab fa-deezer',
			'fa-f55a' => 'fas fa-delete-left',
			'fa-f1a5' => 'fab fa-delicious',
			'fa-f747' => 'fas fa-democrat',
			'fa-f38e' => 'fab fa-deploydog',
			'fa-f38f' => 'fab fa-deskpro',
			'fa-f390' => 'fas fa-desktop',
			'fa-f6cc' => 'fab fa-dev',
			'fa-f1bd' => 'fab fa-deviantart',
			'fa-f655' => 'fas fa-dharmachakra',
			'fa-f790' => 'fab fa-dhl',
			'fa-e476' => 'fas fa-diagram-next',
			'fa-e477' => 'fas fa-diagram-predecessor',
			'fa-f542' => 'fas fa-diagram-project',
			'fa-e47a' => 'fas fa-diagram-successor',
			'fa-f219' => 'fas fa-diamond',
			'fa-f5eb' => 'fas fa-diamond-turn-right',
			'fa-f791' => 'fab fa-diaspora',
			'fa-f522' => 'fas fa-dice',
			'fa-f6cf' => 'fas fa-dice-d20',
			'fa-f6d1' => 'fas fa-dice-d6',
			'fa-f523' => 'fas fa-dice-five',
			'fa-f524' => 'fas fa-dice-four',
			'fa-f525' => 'fas fa-dice-one',
			'fa-f526' => 'fas fa-dice-six',
			'fa-f527' => 'fas fa-dice-three',
			'fa-f528' => 'fas fa-dice-two',
			'fa-f1a6' => 'fab fa-digg',
			'fa-f391' => 'fab fa-digital-ocean',
			'fa-f392' => 'fab fa-discord',
			'fa-f393' => 'fab fa-discourse',
			'fa-f7fa' => 'fas fa-disease',
			'fa-e163' => 'fas fa-display',
			'fa-f529' => 'fas fa-divide',
			'fa-f471' => 'fas fa-dna',
			'fa-f394' => 'fab fa-dochub',
			'fa-f395' => 'fab fa-docker',
			'fa-f6d3' => 'fas fa-dog',
			'fa-24' => 'fas fa-dollar-sign',
			'fa-f472' => 'fas fa-dolly',
			'fa-e169' => 'fas fa-dong-sign',
			'fa-f52a' => 'fas fa-door-closed',
			'fa-f52b' => 'fas fa-door-open',
			'fa-f4ba' => 'fas fa-dove',
			'fa-f422' => 'fas fa-down-left-and-up-right-to-center',
			'fa-f309' => 'fas fa-down-long',
			'fa-f019' => 'fas fa-download',
			'fa-f396' => 'fab fa-draft2digital',
			'fa-f6d5' => 'fas fa-dragon',
			'fa-f5ee' => 'fas fa-draw-polygon',
			'fa-f17d' => 'fab fa-dribbble',
			'fa-f16b' => 'fab fa-dropbox',
			'fa-f043' => 'fas fa-droplet',
			'fa-f5c7' => 'fas fa-droplet-slash',
			'fa-f569' => 'fas fa-drum',
			'fa-f56a' => 'fas fa-drum-steelpan',
			'fa-f6d7' => 'fas fa-drumstick-bite',
			'fa-f1a9' => 'fab fa-drupal',
			'fa-f44b' => 'fas fa-dumbbell',
			'fa-f793' => 'fas fa-dumpster',
			'fa-f794' => 'fas fa-dumpster-fire',
			'fa-f6d9' => 'fas fa-dungeon',
			'fa-f399' => 'fab fa-dyalog',
			'fa-45' => 'fas fa-e',
			'fa-f2a4' => 'fas fa-ear-deaf',
			'fa-f2a2' => 'fas fa-ear-listen',
			'fa-f39a' => 'fab fa-earlybirds',
			'fa-f57c' => 'fas fa-earth-africa',
			'fa-f57d' => 'fas fa-earth-americas',
			'fa-f57e' => 'fas fa-earth-asia',
			'fa-f7a2' => 'fas fa-earth-europe',
			'fa-e47b' => 'fas fa-earth-oceania',
			'fa-f4f4' => 'fab fa-ebay',
			'fa-f282' => 'fab fa-edge',
			'fa-e078' => 'fab fa-edge-legacy',
			'fa-f7fb' => 'fas fa-egg',
			'fa-f052' => 'fas fa-eject',
			'fa-f430' => 'fab fa-elementor',
			'fa-e16d' => 'fas fa-elevator',
			'fa-f141' => 'fas fa-ellipsis',
			'fa-f142' => 'fas fa-ellipsis-vertical',
			'fa-f5f1' => 'fab fa-ello',
			'fa-f423' => 'fab fa-ember',
			'fa-f1d1' => 'fab fa-empire',
			'fa-f0e0' => 'fas fa-envelope',
			'fa-f0e0' => 'far fa-envelope',
			'fa-e4e8' => 'fas fa-envelope-circle-check',
			'fa-f2b6' => 'fas fa-envelope-open',
			'fa-f2b6' => 'far fa-envelope-open',
			'fa-f658' => 'fas fa-envelope-open-text',
			'fa-f674' => 'fas fa-envelopes-bulk',
			'fa-f299' => 'fab fa-envira',
			'fa-3d' => 'fas fa-equals',
			'fa-f12d' => 'fas fa-eraser',
			'fa-f39d' => 'fab fa-erlang',
			'fa-f42e' => 'fab fa-ethereum',
			'fa-f796' => 'fas fa-ethernet',
			'fa-f2d7' => 'fab fa-etsy',
			'fa-f153' => 'fas fa-euro-sign',
			'fa-f839' => 'fab fa-evernote',
			'fa-21' => 'fas fa-exclamation',
			'fa-f065' => 'fas fa-expand',
			'fa-f23e' => 'fab fa-expeditedssl',
			'fa-e4e9' => 'fas fa-explosion',
			'fa-f06e' => 'fas fa-eye',
			'fa-f06e' => 'far fa-eye',
			'fa-f1fb' => 'fas fa-eye-dropper',
			'fa-f2a8' => 'fas fa-eye-low-vision',
			'fa-f070' => 'fas fa-eye-slash',
			'fa-f070' => 'far fa-eye-slash',
			'fa-46' => 'fas fa-f',
			'fa-f556' => 'fas fa-face-angry',
			'fa-f556' => 'far fa-face-angry',
			'fa-f567' => 'fas fa-face-dizzy',
			'fa-f567' => 'far fa-face-dizzy',
			'fa-f579' => 'fas fa-face-flushed',
			'fa-f579' => 'far fa-face-flushed',
			'fa-f119' => 'fas fa-face-frown',
			'fa-f119' => 'far fa-face-frown',
			'fa-f57a' => 'fas fa-face-frown-open',
			'fa-f57a' => 'far fa-face-frown-open',
			'fa-f57f' => 'fas fa-face-grimace',
			'fa-f57f' => 'far fa-face-grimace',
			'fa-f580' => 'fas fa-face-grin',
			'fa-f580' => 'far fa-face-grin',
			'fa-f582' => 'fas fa-face-grin-beam',
			'fa-f582' => 'far fa-face-grin-beam',
			'fa-f583' => 'fas fa-face-grin-beam-sweat',
			'fa-f583' => 'far fa-face-grin-beam-sweat',
			'fa-f584' => 'fas fa-face-grin-hearts',
			'fa-f584' => 'far fa-face-grin-hearts',
			'fa-f585' => 'fas fa-face-grin-squint',
			'fa-f585' => 'far fa-face-grin-squint',
			'fa-f586' => 'fas fa-face-grin-squint-tears',
			'fa-f586' => 'far fa-face-grin-squint-tears',
			'fa-f587' => 'fas fa-face-grin-stars',
			'fa-f587' => 'far fa-face-grin-stars',
			'fa-f588' => 'fas fa-face-grin-tears',
			'fa-f588' => 'far fa-face-grin-tears',
			'fa-f589' => 'fas fa-face-grin-tongue',
			'fa-f589' => 'far fa-face-grin-tongue',
			'fa-f58a' => 'fas fa-face-grin-tongue-squint',
			'fa-f58a' => 'far fa-face-grin-tongue-squint',
			'fa-f58b' => 'fas fa-face-grin-tongue-wink',
			'fa-f58b' => 'far fa-face-grin-tongue-wink',
			'fa-f581' => 'fas fa-face-grin-wide',
			'fa-f581' => 'far fa-face-grin-wide',
			'fa-f58c' => 'fas fa-face-grin-wink',
			'fa-f58c' => 'far fa-face-grin-wink',
			'fa-f596' => 'fas fa-face-kiss',
			'fa-f596' => 'far fa-face-kiss',
			'fa-f597' => 'fas fa-face-kiss-beam',
			'fa-f597' => 'far fa-face-kiss-beam',
			'fa-f598' => 'fas fa-face-kiss-wink-heart',
			'fa-f598' => 'far fa-face-kiss-wink-heart',
			'fa-f599' => 'fas fa-face-laugh',
			'fa-f599' => 'far fa-face-laugh',
			'fa-f59a' => 'fas fa-face-laugh-beam',
			'fa-f59a' => 'far fa-face-laugh-beam',
			'fa-f59b' => 'fas fa-face-laugh-squint',
			'fa-f59b' => 'far fa-face-laugh-squint',
			'fa-f59c' => 'fas fa-face-laugh-wink',
			'fa-f59c' => 'far fa-face-laugh-wink',
			'fa-f11a' => 'fas fa-face-meh',
			'fa-f11a' => 'far fa-face-meh',
			'fa-f5a4' => 'fas fa-face-meh-blank',
			'fa-f5a4' => 'far fa-face-meh-blank',
			'fa-f5a5' => 'fas fa-face-rolling-eyes',
			'fa-f5a5' => 'far fa-face-rolling-eyes',
			'fa-f5b3' => 'fas fa-face-sad-cry',
			'fa-f5b3' => 'far fa-face-sad-cry',
			'fa-f5b4' => 'fas fa-face-sad-tear',
			'fa-f5b4' => 'far fa-face-sad-tear',
			'fa-f118' => 'fas fa-face-smile',
			'fa-f118' => 'far fa-face-smile',
			'fa-f5b8' => 'fas fa-face-smile-beam',
			'fa-f5b8' => 'far fa-face-smile-beam',
			'fa-f4da' => 'fas fa-face-smile-wink',
			'fa-f4da' => 'far fa-face-smile-wink',
			'fa-f5c2' => 'fas fa-face-surprise',
			'fa-f5c2' => 'far fa-face-surprise',
			'fa-f5c8' => 'fas fa-face-tired',
			'fa-f5c8' => 'far fa-face-tired',
			'fa-f09a' => 'fab fa-facebook',
			'fa-f39e' => 'fab fa-facebook-f',
			'fa-f39f' => 'fab fa-facebook-messenger',
			'fa-f863' => 'fas fa-fan',
			'fa-f6dc' => 'fab fa-fantasy-flight-games',
			'fa-e005' => 'fas fa-faucet',
			'fa-e006' => 'fas fa-faucet-drip',
			'fa-f1ac' => 'fas fa-fax',
			'fa-f52d' => 'fas fa-feather',
			'fa-f56b' => 'fas fa-feather-pointed',
			'fa-f797' => 'fab fa-fedex',
			'fa-f798' => 'fab fa-fedora',
			'fa-e4ea' => 'fas fa-ferry',
			'fa-f799' => 'fab fa-figma',
			'fa-f15b' => 'fas fa-file',
			'fa-f15b' => 'far fa-file',
			'fa-f56d' => 'fas fa-file-arrow-down',
			'fa-f574' => 'fas fa-file-arrow-up',
			'fa-f1c7' => 'fas fa-file-audio',
			'fa-f1c7' => 'far fa-file-audio',
			'fa-e5a0' => 'fas fa-file-circle-check',
			'fa-e4eb' => 'fas fa-file-circle-exclamation',
			'fa-e4ed' => 'fas fa-file-circle-minus',
			'fa-e494' => 'fas fa-file-circle-plus',
			'fa-e4ef' => 'fas fa-file-circle-question',
			'fa-e5a1' => 'fas fa-file-circle-xmark',
			'fa-f1c9' => 'fas fa-file-code',
			'fa-f1c9' => 'far fa-file-code',
			'fa-f56c' => 'fas fa-file-contract',
			'fa-f6dd' => 'fas fa-file-csv',
			'fa-f1c3' => 'fas fa-file-excel',
			'fa-f1c3' => 'far fa-file-excel',
			'fa-f56e' => 'fas fa-file-export',
			'fa-f1c5' => 'fas fa-file-image',
			'fa-f1c5' => 'far fa-file-image',
			'fa-f56f' => 'fas fa-file-import',
			'fa-f570' => 'fas fa-file-invoice',
			'fa-f571' => 'fas fa-file-invoice-dollar',
			'fa-f15c' => 'fas fa-file-lines',
			'fa-f15c' => 'far fa-file-lines',
			'fa-f477' => 'fas fa-file-medical',
			'fa-f1c1' => 'fas fa-file-pdf',
			'fa-f1c1' => 'far fa-file-pdf',
			'fa-f31c' => 'fas fa-file-pen',
			'fa-f1c4' => 'fas fa-file-powerpoint',
			'fa-f1c4' => 'far fa-file-powerpoint',
			'fa-f572' => 'fas fa-file-prescription',
			'fa-e4f0' => 'fas fa-file-shield',
			'fa-f573' => 'fas fa-file-signature',
			'fa-f1c8' => 'fas fa-file-video',
			'fa-f1c8' => 'far fa-file-video',
			'fa-f478' => 'fas fa-file-waveform',
			'fa-f1c2' => 'fas fa-file-word',
			'fa-f1c2' => 'far fa-file-word',
			'fa-f1c6' => 'fas fa-file-zipper',
			'fa-f1c6' => 'far fa-file-zipper',
			'fa-f575' => 'fas fa-fill',
			'fa-f576' => 'fas fa-fill-drip',
			'fa-f008' => 'fas fa-film',
			'fa-f0b0' => 'fas fa-filter',
			'fa-f662' => 'fas fa-filter-circle-dollar',
			'fa-e17b' => 'fas fa-filter-circle-xmark',
			'fa-f577' => 'fas fa-fingerprint',
			'fa-f06d' => 'fas fa-fire',
			'fa-e4f1' => 'fas fa-fire-burner',
			'fa-f134' => 'fas fa-fire-extinguisher',
			'fa-f7e4' => 'fas fa-fire-flame-curved',
			'fa-f46a' => 'fas fa-fire-flame-simple',
			'fa-f269' => 'fab fa-firefox',
			'fa-e007' => 'fab fa-firefox-browser',
			'fa-f2b0' => 'fab fa-first-order',
			'fa-f50a' => 'fab fa-first-order-alt',
			'fa-f3a1' => 'fab fa-firstdraft',
			'fa-f578' => 'fas fa-fish',
			'fa-e4f2' => 'fas fa-fish-fins',
			'fa-f024' => 'fas fa-flag',
			'fa-f024' => 'far fa-flag',
			'fa-f11e' => 'fas fa-flag-checkered',
			'fa-f74d' => 'fas fa-flag-usa',
			'fa-f0c3' => 'fas fa-flask',
			'fa-e4f3' => 'fas fa-flask-vial',
			'fa-f16e' => 'fab fa-flickr',
			'fa-f44d' => 'fab fa-flipboard',
			'fa-f0c7' => 'fas fa-floppy-disk',
			'fa-f0c7' => 'far fa-floppy-disk',
			'fa-e184' => 'fas fa-florin-sign',
			'fa-f417' => 'fab fa-fly',
			'fa-f07b' => 'fas fa-folder',
			'fa-f07b' => 'far fa-folder',
			'fa-e185' => 'fas fa-folder-closed',
			'fa-e185' => 'far fa-folder-closed',
			'fa-f65d' => 'fas fa-folder-minus',
			'fa-f07c' => 'fas fa-folder-open',
			'fa-f07c' => 'far fa-folder-open',
			'fa-f65e' => 'fas fa-folder-plus',
			'fa-f802' => 'fas fa-folder-tree',
			'fa-f031' => 'fas fa-font',
			'fa-f2b4' => 'fas fa-font-awesome',
			'fa-f2b4' => 'far fa-font-awesome',
			'fa-f2b4' => 'fab fa-font-awesome',
			'fa-f280' => 'fab fa-fonticons',
			'fa-f3a2' => 'fab fa-fonticons-fi',
			'fa-f44e' => 'fas fa-football',
			'fa-f286' => 'fab fa-fort-awesome',
			'fa-f3a3' => 'fab fa-fort-awesome-alt',
			'fa-f211' => 'fab fa-forumbee',
			'fa-f04e' => 'fas fa-forward',
			'fa-f050' => 'fas fa-forward-fast',
			'fa-f051' => 'fas fa-forward-step',
			'fa-f180' => 'fab fa-foursquare',
			'fa-e18f' => 'fas fa-franc-sign',
			'fa-f2c5' => 'fab fa-free-code-camp',
			'fa-f3a4' => 'fab fa-freebsd',
			'fa-f52e' => 'fas fa-frog',
			'fa-f50b' => 'fab fa-fulcrum',
			'fa-f1e3' => 'fas fa-futbol',
			'fa-f1e3' => 'far fa-futbol',
			'fa-47' => 'fas fa-g',
			'fa-f50c' => 'fab fa-galactic-republic',
			'fa-f50d' => 'fab fa-galactic-senate',
			'fa-f11b' => 'fas fa-gamepad',
			'fa-f52f' => 'fas fa-gas-pump',
			'fa-f624' => 'fas fa-gauge',
			'fa-f625' => 'fas fa-gauge-high',
			'fa-f629' => 'fas fa-gauge-simple',
			'fa-f62a' => 'fas fa-gauge-simple-high',
			'fa-f0e3' => 'fas fa-gavel',
			'fa-f013' => 'fas fa-gear',
			'fa-f085' => 'fas fa-gears',
			'fa-f3a5' => 'fas fa-gem',
			'fa-f3a5' => 'far fa-gem',
			'fa-f22d' => 'fas fa-genderless',
			'fa-f265' => 'fab fa-get-pocket',
			'fa-f260' => 'fab fa-gg',
			'fa-f261' => 'fab fa-gg-circle',
			'fa-f6e2' => 'fas fa-ghost',
			'fa-f06b' => 'fas fa-gift',
			'fa-f79c' => 'fas fa-gifts',
			'fa-f1d3' => 'fab fa-git',
			'fa-f841' => 'fab fa-git-alt',
			'fa-f09b' => 'fab fa-github',
			'fa-f113' => 'fab fa-github-alt',
			'fa-f3a6' => 'fab fa-gitkraken',
			'fa-f296' => 'fab fa-gitlab',
			'fa-f426' => 'fab fa-gitter',
			'fa-e4f4' => 'fas fa-glass-water',
			'fa-e4f5' => 'fas fa-glass-water-droplet',
			'fa-f530' => 'fas fa-glasses',
			'fa-f2a5' => 'fab fa-glide',
			'fa-f2a6' => 'fab fa-glide-g',
			'fa-f0ac' => 'fas fa-globe',
			'fa-f3a7' => 'fab fa-gofore',
			'fa-e40f' => 'fab fa-golang',
			'fa-f450' => 'fas fa-golf-ball-tee',
			'fa-f3a8' => 'fab fa-goodreads',
			'fa-f3a9' => 'fab fa-goodreads-g',
			'fa-f1a0' => 'fab fa-google',
			'fa-f3aa' => 'fab fa-google-drive',
			'fa-e079' => 'fab fa-google-pay',
			'fa-f3ab' => 'fab fa-google-play',
			'fa-f2b3' => 'fab fa-google-plus',
			'fa-f0d5' => 'fab fa-google-plus-g',
			'fa-f1ee' => 'fab fa-google-wallet',
			'fa-f664' => 'fas fa-gopuram',
			'fa-f19d' => 'fas fa-graduation-cap',
			'fa-f184' => 'fab fa-gratipay',
			'fa-f2d6' => 'fab fa-grav',
			'fa-3e' => 'fas fa-greater-than',
			'fa-f532' => 'fas fa-greater-than-equal',
			'fa-f58d' => 'fas fa-grip',
			'fa-f7a4' => 'fas fa-grip-lines',
			'fa-f7a5' => 'fas fa-grip-lines-vertical',
			'fa-f58e' => 'fas fa-grip-vertical',
			'fa-f3ac' => 'fab fa-gripfire',
			'fa-e4f6' => 'fas fa-group-arrows-rotate',
			'fa-f3ad' => 'fab fa-grunt',
			'fa-e19a' => 'fas fa-guarani-sign',
			'fa-e07e' => 'fab fa-guilded',
			'fa-f7a6' => 'fas fa-guitar',
			'fa-f3ae' => 'fab fa-gulp',
			'fa-e19b' => 'fas fa-gun',
			'fa-48' => 'fas fa-h',
			'fa-f1d4' => 'fab fa-hacker-news',
			'fa-f5f7' => 'fab fa-hackerrank',
			'fa-f6e3' => 'fas fa-hammer',
			'fa-f665' => 'fas fa-hamsa',
			'fa-f256' => 'fas fa-hand',
			'fa-f256' => 'far fa-hand',
			'fa-f255' => 'fas fa-hand-back-fist',
			'fa-f255' => 'far fa-hand-back-fist',
			'fa-f461' => 'fas fa-hand-dots',
			'fa-f6de' => 'fas fa-hand-fist',
			'fa-f4bd' => 'fas fa-hand-holding',
			'fa-f4c0' => 'fas fa-hand-holding-dollar',
			'fa-f4c1' => 'fas fa-hand-holding-droplet',
			'fa-e4f7' => 'fas fa-hand-holding-hand',
			'fa-f4be' => 'fas fa-hand-holding-heart',
			'fa-e05c' => 'fas fa-hand-holding-medical',
			'fa-f258' => 'fas fa-hand-lizard',
			'fa-f258' => 'far fa-hand-lizard',
			'fa-f806' => 'fas fa-hand-middle-finger',
			'fa-f25b' => 'fas fa-hand-peace',
			'fa-f25b' => 'far fa-hand-peace',
			'fa-f0a7' => 'fas fa-hand-point-down',
			'fa-f0a7' => 'far fa-hand-point-down',
			'fa-f0a5' => 'fas fa-hand-point-left',
			'fa-f0a5' => 'far fa-hand-point-left',
			'fa-f0a4' => 'fas fa-hand-point-right',
			'fa-f0a4' => 'far fa-hand-point-right',
			'fa-f0a6' => 'fas fa-hand-point-up',
			'fa-f0a6' => 'far fa-hand-point-up',
			'fa-f25a' => 'fas fa-hand-pointer',
			'fa-f25a' => 'far fa-hand-pointer',
			'fa-f257' => 'fas fa-hand-scissors',
			'fa-f257' => 'far fa-hand-scissors',
			'fa-e05d' => 'fas fa-hand-sparkles',
			'fa-f259' => 'fas fa-hand-spock',
			'fa-f259' => 'far fa-hand-spock',
			'fa-e4f8' => 'fas fa-handcuffs',
			'fa-f2a7' => 'fas fa-hands',
			'fa-f2a3' => 'fas fa-hands-asl-interpreting',
			'fa-e4f9' => 'fas fa-hands-bound',
			'fa-e05e' => 'fas fa-hands-bubbles',
			'fa-e1a8' => 'fas fa-hands-clapping',
			'fa-f4c2' => 'fas fa-hands-holding',
			'fa-e4fa' => 'fas fa-hands-holding-child',
			'fa-e4fb' => 'fas fa-hands-holding-circle',
			'fa-f684' => 'fas fa-hands-praying',
			'fa-f2b5' => 'fas fa-handshake',
			'fa-f2b5' => 'far fa-handshake',
			'fa-f4c4' => 'fas fa-handshake-angle',
			'fa-f4c6' => 'fas fa-handshake-simple',
			'fa-e05f' => 'fas fa-handshake-simple-slash',
			'fa-e060' => 'fas fa-handshake-slash',
			'fa-f6e6' => 'fas fa-hanukiah',
			'fa-f0a0' => 'fas fa-hard-drive',
			'fa-f0a0' => 'far fa-hard-drive',
			'fa-e499' => 'fab fa-hashnode',
			'fa-23' => 'fas fa-hashtag',
			'fa-f8c0' => 'fas fa-hat-cowboy',
			'fa-f8c1' => 'fas fa-hat-cowboy-side',
			'fa-f6e8' => 'fas fa-hat-wizard',
			'fa-e061' => 'fas fa-head-side-cough',
			'fa-e062' => 'fas fa-head-side-cough-slash',
			'fa-e063' => 'fas fa-head-side-mask',
			'fa-e064' => 'fas fa-head-side-virus',
			'fa-f1dc' => 'fas fa-heading',
			'fa-f025' => 'fas fa-headphones',
			'fa-f58f' => 'fas fa-headphones-simple',
			'fa-f590' => 'fas fa-headset',
			'fa-f004' => 'fas fa-heart',
			'fa-f004' => 'far fa-heart',
			'fa-e4fc' => 'fas fa-heart-circle-bolt',
			'fa-e4fd' => 'fas fa-heart-circle-check',
			'fa-e4fe' => 'fas fa-heart-circle-exclamation',
			'fa-e4ff' => 'fas fa-heart-circle-minus',
			'fa-e500' => 'fas fa-heart-circle-plus',
			'fa-e501' => 'fas fa-heart-circle-xmark',
			'fa-f7a9' => 'fas fa-heart-crack',
			'fa-f21e' => 'fas fa-heart-pulse',
			'fa-f533' => 'fas fa-helicopter',
			'fa-e502' => 'fas fa-helicopter-symbol',
			'fa-f807' => 'fas fa-helmet-safety',
			'fa-e503' => 'fas fa-helmet-un',
			'fa-f591' => 'fas fa-highlighter',
			'fa-e507' => 'fas fa-hill-avalanche',
			'fa-e508' => 'fas fa-hill-rockslide',
			'fa-f6ed' => 'fas fa-hippo',
			'fa-f452' => 'fab fa-hips',
			'fa-f3b0' => 'fab fa-hire-a-helper',
			'fa-e07f' => 'fab fa-hive',
			'fa-f453' => 'fas fa-hockey-puck',
			'fa-f7aa' => 'fas fa-holly-berry',
			'fa-f427' => 'fab fa-hooli',
			'fa-f592' => 'fab fa-hornbill',
			'fa-f6f0' => 'fas fa-horse',
			'fa-f7ab' => 'fas fa-horse-head',
			'fa-f0f8' => 'fas fa-hospital',
			'fa-f0f8' => 'far fa-hospital',
			'fa-f80d' => 'fas fa-hospital-user',
			'fa-f593' => 'fas fa-hot-tub-person',
			'fa-f80f' => 'fas fa-hotdog',
			'fa-f594' => 'fas fa-hotel',
			'fa-f3b1' => 'fab fa-hotjar',
			'fa-f254' => 'fas fa-hourglass',
			'fa-f254' => 'far fa-hourglass',
			'fa-f253' => 'fas fa-hourglass-end',
			'fa-f252' => 'fas fa-hourglass-half',
			'fa-f252' => 'far fa-hourglass-half',
			'fa-f251' => 'fas fa-hourglass-start',
			'fa-f015' => 'fas fa-house',
			'fa-e3af' => 'fas fa-house-chimney',
			'fa-f6f1' => 'fas fa-house-chimney-crack',
			'fa-f7f2' => 'fas fa-house-chimney-medical',
			'fa-e065' => 'fas fa-house-chimney-user',
			'fa-e00d' => 'fas fa-house-chimney-window',
			'fa-e509' => 'fas fa-house-circle-check',
			'fa-e50a' => 'fas fa-house-circle-exclamation',
			'fa-e50b' => 'fas fa-house-circle-xmark',
			'fa-e3b1' => 'fas fa-house-crack',
			'fa-e50c' => 'fas fa-house-fire',
			'fa-e50d' => 'fas fa-house-flag',
			'fa-e50e' => 'fas fa-house-flood-water',
			'fa-e50f' => 'fas fa-house-flood-water-circle-arrow-right',
			'fa-e066' => 'fas fa-house-laptop',
			'fa-e510' => 'fas fa-house-lock',
			'fa-e3b2' => 'fas fa-house-medical',
			'fa-e511' => 'fas fa-house-medical-circle-check',
			'fa-e512' => 'fas fa-house-medical-circle-exclamation',
			'fa-e513' => 'fas fa-house-medical-circle-xmark',
			'fa-e514' => 'fas fa-house-medical-flag',
			'fa-e012' => 'fas fa-house-signal',
			'fa-e515' => 'fas fa-house-tsunami',
			'fa-e1b0' => 'fas fa-house-user',
			'fa-f27c' => 'fab fa-houzz',
			'fa-f6f2' => 'fas fa-hryvnia-sign',
			'fa-f13b' => 'fab fa-html5',
			'fa-f3b2' => 'fab fa-hubspot',
			'fa-f751' => 'fas fa-hurricane',
			'fa-49' => 'fas fa-i',
			'fa-f246' => 'fas fa-i-cursor',
			'fa-f810' => 'fas fa-ice-cream',
			'fa-f7ad' => 'fas fa-icicles',
			'fa-f86d' => 'fas fa-icons',
			'fa-f2c1' => 'fas fa-id-badge',
			'fa-f2c1' => 'far fa-id-badge',
			'fa-f2c2' => 'fas fa-id-card',
			'fa-f2c2' => 'far fa-id-card',
			'fa-f47f' => 'fas fa-id-card-clip',
			'fa-e013' => 'fab fa-ideal',
			'fa-f7ae' => 'fas fa-igloo',
			'fa-f03e' => 'fas fa-image',
			'fa-f03e' => 'far fa-image',
			'fa-f3e0' => 'fas fa-image-portrait',
			'fa-f302' => 'fas fa-images',
			'fa-f302' => 'far fa-images',
			'fa-f2d8' => 'fab fa-imdb',
			'fa-f01c' => 'fas fa-inbox',
			'fa-f03c' => 'fas fa-indent',
			'fa-e1bc' => 'fas fa-indian-rupee-sign',
			'fa-f275' => 'fas fa-industry',
			'fa-f534' => 'fas fa-infinity',
			'fa-f129' => 'fas fa-info',
			'fa-f16d' => 'fab fa-instagram',
			'fa-e081' => 'fab fa-instalod',
			'fa-f7af' => 'fab fa-intercom',
			'fa-f26b' => 'fab fa-internet-explorer',
			'fa-f7b0' => 'fab fa-invision',
			'fa-f208' => 'fab fa-ioxhost',
			'fa-f033' => 'fas fa-italic',
			'fa-f83a' => 'fab fa-itch-io',
			'fa-f3b4' => 'fab fa-itunes',
			'fa-f3b5' => 'fab fa-itunes-note',
			'fa-4a' => 'fas fa-j',
			'fa-e516' => 'fas fa-jar',
			'fa-e517' => 'fas fa-jar-wheat',
			'fa-f4e4' => 'fab fa-java',
			'fa-f669' => 'fas fa-jedi',
			'fa-f50e' => 'fab fa-jedi-order',
			'fa-f3b6' => 'fab fa-jenkins',
			'fa-f0fb' => 'fas fa-jet-fighter',
			'fa-e518' => 'fas fa-jet-fighter-up',
			'fa-f7b1' => 'fab fa-jira',
			'fa-f3b7' => 'fab fa-joget',
			'fa-f595' => 'fas fa-joint',
			'fa-f1aa' => 'fab fa-joomla',
			'fa-f3b8' => 'fab fa-js',
			'fa-f1cc' => 'fab fa-jsfiddle',
			'fa-e519' => 'fas fa-jug-detergent',
			'fa-4b' => 'fas fa-k',
			'fa-f66b' => 'fas fa-kaaba',
			'fa-f5fa' => 'fab fa-kaggle',
			'fa-f084' => 'fas fa-key',
			'fa-f4f5' => 'fab fa-keybase',
			'fa-f11c' => 'fas fa-keyboard',
			'fa-f11c' => 'far fa-keyboard',
			'fa-f3ba' => 'fab fa-keycdn',
			'fa-f66d' => 'fas fa-khanda',
			'fa-f3bb' => 'fab fa-kickstarter',
			'fa-f3bc' => 'fab fa-kickstarter-k',
			'fa-e1c4' => 'fas fa-kip-sign',
			'fa-f479' => 'fas fa-kit-medical',
			'fa-e51a' => 'fas fa-kitchen-set',
			'fa-f535' => 'fas fa-kiwi-bird',
			'fa-f42f' => 'fab fa-korvue',
			'fa-4c' => 'fas fa-l',
			'fa-e51b' => 'fas fa-land-mine-on',
			'fa-f66f' => 'fas fa-landmark',
			'fa-f752' => 'fas fa-landmark-dome',
			'fa-e51c' => 'fas fa-landmark-flag',
			'fa-f1ab' => 'fas fa-language',
			'fa-f109' => 'fas fa-laptop',
			'fa-f5fc' => 'fas fa-laptop-code',
			'fa-e51d' => 'fas fa-laptop-file',
			'fa-f812' => 'fas fa-laptop-medical',
			'fa-f3bd' => 'fab fa-laravel',
			'fa-e1c8' => 'fas fa-lari-sign',
			'fa-f202' => 'fab fa-lastfm',
			'fa-f5fd' => 'fas fa-layer-group',
			'fa-f06c' => 'fas fa-leaf',
			'fa-f212' => 'fab fa-leanpub',
			'fa-f30a' => 'fas fa-left-long',
			'fa-f337' => 'fas fa-left-right',
			'fa-f094' => 'fas fa-lemon',
			'fa-f094' => 'far fa-lemon',
			'fa-f41d' => 'fab fa-less',
			'fa-3c' => 'fas fa-less-than',
			'fa-f537' => 'fas fa-less-than-equal',
			'fa-f1cd' => 'fas fa-life-ring',
			'fa-f1cd' => 'far fa-life-ring',
			'fa-f0eb' => 'fas fa-lightbulb',
			'fa-f0eb' => 'far fa-lightbulb',
			'fa-f3c0' => 'fab fa-line',
			'fa-e51e' => 'fas fa-lines-leaning',
			'fa-f0c1' => 'fas fa-link',
			'fa-f127' => 'fas fa-link-slash',
			'fa-f08c' => 'fab fa-linkedin',
			'fa-f0e1' => 'fab fa-linkedin-in',
			'fa-f2b8' => 'fab fa-linode',
			'fa-f17c' => 'fab fa-linux',
			'fa-f195' => 'fas fa-lira-sign',
			'fa-f03a' => 'fas fa-list',
			'fa-f0ae' => 'fas fa-list-check',
			'fa-f0cb' => 'fas fa-list-ol',
			'fa-f0ca' => 'fas fa-list-ul',
			'fa-e1d3' => 'fas fa-litecoin-sign',
			'fa-f124' => 'fas fa-location-arrow',
			'fa-f601' => 'fas fa-location-crosshairs',
			'fa-f3c5' => 'fas fa-location-dot',
			'fa-f041' => 'fas fa-location-pin',
			'fa-e51f' => 'fas fa-location-pin-lock',
			'fa-f023' => 'fas fa-lock',
			'fa-f3c1' => 'fas fa-lock-open',
			'fa-e520' => 'fas fa-locust',
			'fa-f604' => 'fas fa-lungs',
			'fa-e067' => 'fas fa-lungs-virus',
			'fa-f3c3' => 'fab fa-lyft',
			'fa-4d' => 'fas fa-m',
			'fa-f3c4' => 'fab fa-magento',
			'fa-f076' => 'fas fa-magnet',
			'fa-f002' => 'fas fa-magnifying-glass',
			'fa-e521' => 'fas fa-magnifying-glass-arrow-right',
			'fa-e522' => 'fas fa-magnifying-glass-chart',
			'fa-f688' => 'fas fa-magnifying-glass-dollar',
			'fa-f689' => 'fas fa-magnifying-glass-location',
			'fa-f010' => 'fas fa-magnifying-glass-minus',
			'fa-f00e' => 'fas fa-magnifying-glass-plus',
			'fa-f59e' => 'fab fa-mailchimp',
			'fa-e1d5' => 'fas fa-manat-sign',
			'fa-f50f' => 'fab fa-mandalorian',
			'fa-f279' => 'fas fa-map',
			'fa-f279' => 'far fa-map',
			'fa-f59f' => 'fas fa-map-location',
			'fa-f5a0' => 'fas fa-map-location-dot',
			'fa-f276' => 'fas fa-map-pin',
			'fa-f60f' => 'fab fa-markdown',
			'fa-f5a1' => 'fas fa-marker',
			'fa-f222' => 'fas fa-mars',
			'fa-f224' => 'fas fa-mars-and-venus',
			'fa-e523' => 'fas fa-mars-and-venus-burst',
			'fa-f227' => 'fas fa-mars-double',
			'fa-f229' => 'fas fa-mars-stroke',
			'fa-f22b' => 'fas fa-mars-stroke-right',
			'fa-f22a' => 'fas fa-mars-stroke-up',
			'fa-f57b' => 'fas fa-martini-glass',
			'fa-f561' => 'fas fa-martini-glass-citrus',
			'fa-f000' => 'fas fa-martini-glass-empty',
			'fa-f6fa' => 'fas fa-mask',
			'fa-e1d7' => 'fas fa-mask-face',
			'fa-e524' => 'fas fa-mask-ventilator',
			'fa-f630' => 'fas fa-masks-theater',
			'fa-f4f6' => 'fab fa-mastodon',
			'fa-e525' => 'fas fa-mattress-pillow',
			'fa-f136' => 'fab fa-maxcdn',
			'fa-f31e' => 'fas fa-maximize',
			'fa-f8ca' => 'fab fa-mdb',
			'fa-f5a2' => 'fas fa-medal',
			'fa-f3c6' => 'fab fa-medapps',
			'fa-f23a' => 'fab fa-medium',
			'fa-f3c8' => 'fab fa-medrt',
			'fa-f2e0' => 'fab fa-meetup',
			'fa-f5a3' => 'fab fa-megaport',
			'fa-f538' => 'fas fa-memory',
			'fa-f7b3' => 'fab fa-mendeley',
			'fa-f676' => 'fas fa-menorah',
			'fa-f223' => 'fas fa-mercury',
			'fa-f27a' => 'fas fa-message',
			'fa-f27a' => 'far fa-message',
			'fa-e49b' => 'fab fa-meta',
			'fa-f753' => 'fas fa-meteor',
			'fa-e01a' => 'fab fa-microblog',
			'fa-f2db' => 'fas fa-microchip',
			'fa-f130' => 'fas fa-microphone',
			'fa-f3c9' => 'fas fa-microphone-lines',
			'fa-f539' => 'fas fa-microphone-lines-slash',
			'fa-f131' => 'fas fa-microphone-slash',
			'fa-f610' => 'fas fa-microscope',
			'fa-f3ca' => 'fab fa-microsoft',
			'fa-e1ed' => 'fas fa-mill-sign',
			'fa-f78c' => 'fas fa-minimize',
			'fa-f068' => 'fas fa-minus',
			'fa-f7b5' => 'fas fa-mitten',
			'fa-f3cb' => 'fab fa-mix',
			'fa-f289' => 'fab fa-mixcloud',
			'fa-e056' => 'fab fa-mixer',
			'fa-f3cc' => 'fab fa-mizuni',
			'fa-f3ce' => 'fas fa-mobile',
			'fa-f10b' => 'fas fa-mobile-button',
			'fa-e527' => 'fas fa-mobile-retro',
			'fa-f3cf' => 'fas fa-mobile-screen',
			'fa-f3cd' => 'fas fa-mobile-screen-button',
			'fa-f285' => 'fab fa-modx',
			'fa-f3d0' => 'fab fa-monero',
			'fa-f0d6' => 'fas fa-money-bill',
			'fa-f3d1' => 'fas fa-money-bill-1',
			'fa-f3d1' => 'far fa-money-bill-1',
			'fa-f53b' => 'fas fa-money-bill-1-wave',
			'fa-e528' => 'fas fa-money-bill-transfer',
			'fa-e529' => 'fas fa-money-bill-trend-up',
			'fa-f53a' => 'fas fa-money-bill-wave',
			'fa-e52a' => 'fas fa-money-bill-wheat',
			'fa-e1f3' => 'fas fa-money-bills',
			'fa-f53c' => 'fas fa-money-check',
			'fa-f53d' => 'fas fa-money-check-dollar',
			'fa-f5a6' => 'fas fa-monument',
			'fa-f186' => 'fas fa-moon',
			'fa-f186' => 'far fa-moon',
			'fa-f5a7' => 'fas fa-mortar-pestle',
			'fa-f678' => 'fas fa-mosque',
			'fa-e52b' => 'fas fa-mosquito',
			'fa-e52c' => 'fas fa-mosquito-net',
			'fa-f21c' => 'fas fa-motorcycle',
			'fa-e52d' => 'fas fa-mound',
			'fa-f6fc' => 'fas fa-mountain',
			'fa-e52e' => 'fas fa-mountain-city',
			'fa-e52f' => 'fas fa-mountain-sun',
			'fa-f7b6' => 'fas fa-mug-hot',
			'fa-f0f4' => 'fas fa-mug-saucer',
			'fa-f001' => 'fas fa-music',
			'fa-4e' => 'fas fa-n',
			'fa-e1f6' => 'fas fa-naira-sign',
			'fa-f3d2' => 'fab fa-napster',
			'fa-f612' => 'fab fa-neos',
			'fa-f6ff' => 'fas fa-network-wired',
			'fa-f22c' => 'fas fa-neuter',
			'fa-f1ea' => 'fas fa-newspaper',
			'fa-f1ea' => 'far fa-newspaper',
			'fa-e530' => 'fab fa-nfc-directional',
			'fa-e531' => 'fab fa-nfc-symbol',
			'fa-f5a8' => 'fab fa-nimblr',
			'fa-f419' => 'fab fa-node',
			'fa-f3d3' => 'fab fa-node-js',
			'fa-f53e' => 'fas fa-not-equal',
			'fa-e1fe' => 'fas fa-notdef',
			'fa-f249' => 'fas fa-note-sticky',
			'fa-f249' => 'far fa-note-sticky',
			'fa-f481' => 'fas fa-notes-medical',
			'fa-f3d4' => 'fab fa-npm',
			'fa-f3d5' => 'fab fa-ns8',
			'fa-f3d6' => 'fab fa-nutritionix',
			'fa-4f' => 'fas fa-o',
			'fa-f247' => 'fas fa-object-group',
			'fa-f247' => 'far fa-object-group',
			'fa-f248' => 'fas fa-object-ungroup',
			'fa-f248' => 'far fa-object-ungroup',
			'fa-e082' => 'fab fa-octopus-deploy',
			'fa-f263' => 'fab fa-odnoklassniki',
			'fa-e5c6' => 'fab fa-odysee',
			'fa-f613' => 'fas fa-oil-can',
			'fa-e532' => 'fas fa-oil-well',
			'fa-f510' => 'fab fa-old-republic',
			'fa-f679' => 'fas fa-om',
			'fa-f23d' => 'fab fa-opencart',
			'fa-f19b' => 'fab fa-openid',
			'fa-f26a' => 'fab fa-opera',
			'fa-f23c' => 'fab fa-optin-monster',
			'fa-f8d2' => 'fab fa-orcid',
			'fa-f41a' => 'fab fa-osi',
			'fa-f700' => 'fas fa-otter',
			'fa-f03b' => 'fas fa-outdent',
			'fa-50' => 'fas fa-p',
			'fa-e4a0' => 'fab fa-padlet',
			'fa-f3d7' => 'fab fa-page4',
			'fa-f18c' => 'fab fa-pagelines',
			'fa-f815' => 'fas fa-pager',
			'fa-f5aa' => 'fas fa-paint-roller',
			'fa-f1fc' => 'fas fa-paintbrush',
			'fa-f53f' => 'fas fa-palette',
			'fa-f3d8' => 'fab fa-palfed',
			'fa-f482' => 'fas fa-pallet',
			'fa-e209' => 'fas fa-panorama',
			'fa-f1d8' => 'fas fa-paper-plane',
			'fa-f1d8' => 'far fa-paper-plane',
			'fa-f0c6' => 'fas fa-paperclip',
			'fa-f4cd' => 'fas fa-parachute-box',
			'fa-f1dd' => 'fas fa-paragraph',
			'fa-f5ab' => 'fas fa-passport',
			'fa-f0ea' => 'fas fa-paste',
			'fa-f0ea' => 'far fa-paste',
			'fa-f3d9' => 'fab fa-patreon',
			'fa-f04c' => 'fas fa-pause',
			'fa-f1b0' => 'fas fa-paw',
			'fa-f1ed' => 'fab fa-paypal',
			'fa-f67c' => 'fas fa-peace',
			'fa-f304' => 'fas fa-pen',
			'fa-f305' => 'fas fa-pen-clip',
			'fa-f5ac' => 'fas fa-pen-fancy',
			'fa-f5ad' => 'fas fa-pen-nib',
			'fa-f5ae' => 'fas fa-pen-ruler',
			'fa-f044' => 'fas fa-pen-to-square',
			'fa-f044' => 'far fa-pen-to-square',
			'fa-f303' => 'fas fa-pencil',
			'fa-e068' => 'fas fa-people-arrows',
			'fa-f4ce' => 'fas fa-people-carry-box',
			'fa-e533' => 'fas fa-people-group',
			'fa-e534' => 'fas fa-people-line',
			'fa-e535' => 'fas fa-people-pulling',
			'fa-e536' => 'fas fa-people-robbery',
			'fa-e537' => 'fas fa-people-roof',
			'fa-f816' => 'fas fa-pepper-hot',
			'fa-e083' => 'fab fa-perbyte',
			'fa-25' => 'fas fa-percent',
			'fa-f3da' => 'fab fa-periscope',
			'fa-f183' => 'fas fa-person',
			'fa-e538' => 'fas fa-person-arrow-down-to-line',
			'fa-e539' => 'fas fa-person-arrow-up-from-line',
			'fa-f84a' => 'fas fa-person-biking',
			'fa-f756' => 'fas fa-person-booth',
			'fa-e53a' => 'fas fa-person-breastfeeding',
			'fa-e53b' => 'fas fa-person-burst',
			'fa-e53c' => 'fas fa-person-cane',
			'fa-e53d' => 'fas fa-person-chalkboard',
			'fa-e53e' => 'fas fa-person-circle-check',
			'fa-e53f' => 'fas fa-person-circle-exclamation',
			'fa-e540' => 'fas fa-person-circle-minus',
			'fa-e541' => 'fas fa-person-circle-plus',
			'fa-e542' => 'fas fa-person-circle-question',
			'fa-e543' => 'fas fa-person-circle-xmark',
			'fa-f85e' => 'fas fa-person-digging',
			'fa-f470' => 'fas fa-person-dots-from-line',
			'fa-f182' => 'fas fa-person-dress',
			'fa-e544' => 'fas fa-person-dress-burst',
			'fa-e545' => 'fas fa-person-drowning',
			'fa-e546' => 'fas fa-person-falling',
			'fa-e547' => 'fas fa-person-falling-burst',
			'fa-e548' => 'fas fa-person-half-dress',
			'fa-e549' => 'fas fa-person-harassing',
			'fa-f6ec' => 'fas fa-person-hiking',
			'fa-e54a' => 'fas fa-person-military-pointing',
			'fa-e54b' => 'fas fa-person-military-rifle',
			'fa-e54c' => 'fas fa-person-military-to-person',
			'fa-f683' => 'fas fa-person-praying',
			'fa-e31e' => 'fas fa-person-pregnant',
			'fa-e54d' => 'fas fa-person-rays',
			'fa-e54e' => 'fas fa-person-rifle',
			'fa-f70c' => 'fas fa-person-running',
			'fa-e54f' => 'fas fa-person-shelter',
			'fa-f7c5' => 'fas fa-person-skating',
			'fa-f7c9' => 'fas fa-person-skiing',
			'fa-f7ca' => 'fas fa-person-skiing-nordic',
			'fa-f7ce' => 'fas fa-person-snowboarding',
			'fa-f5c4' => 'fas fa-person-swimming',
			'fa-e5a9' => 'fas fa-person-through-window',
			'fa-f554' => 'fas fa-person-walking',
			'fa-e551' => 'fas fa-person-walking-arrow-loop-left',
			'fa-e552' => 'fas fa-person-walking-arrow-right',
			'fa-e553' => 'fas fa-person-walking-dashed-line-arrow-right',
			'fa-e554' => 'fas fa-person-walking-luggage',
			'fa-f29d' => 'fas fa-person-walking-with-cane',
			'fa-e221' => 'fas fa-peseta-sign',
			'fa-e222' => 'fas fa-peso-sign',
			'fa-f3db' => 'fab fa-phabricator',
			'fa-f3dc' => 'fab fa-phoenix-framework',
			'fa-f511' => 'fab fa-phoenix-squadron',
			'fa-f095' => 'fas fa-phone',
			'fa-f879' => 'fas fa-phone-flip',
			'fa-f3dd' => 'fas fa-phone-slash',
			'fa-f2a0' => 'fas fa-phone-volume',
			'fa-f87c' => 'fas fa-photo-film',
			'fa-f457' => 'fab fa-php',
			'fa-f2ae' => 'fab fa-pied-piper',
			'fa-f1a8' => 'fab fa-pied-piper-alt',
			'fa-f4e5' => 'fab fa-pied-piper-hat',
			'fa-f1a7' => 'fab fa-pied-piper-pp',
			'fa-f4d3' => 'fas fa-piggy-bank',
			'fa-f484' => 'fas fa-pills',
			'fa-f0d2' => 'fab fa-pinterest',
			'fa-f231' => 'fab fa-pinterest-p',
			'fa-e43a' => 'fab fa-pix',
			'fa-f818' => 'fas fa-pizza-slice',
			'fa-f67f' => 'fas fa-place-of-worship',
			'fa-f072' => 'fas fa-plane',
			'fa-f5af' => 'fas fa-plane-arrival',
			'fa-e555' => 'fas fa-plane-circle-check',
			'fa-e556' => 'fas fa-plane-circle-exclamation',
			'fa-e557' => 'fas fa-plane-circle-xmark',
			'fa-f5b0' => 'fas fa-plane-departure',
			'fa-e558' => 'fas fa-plane-lock',
			'fa-e069' => 'fas fa-plane-slash',
			'fa-e22d' => 'fas fa-plane-up',
			'fa-e5aa' => 'fas fa-plant-wilt',
			'fa-e55a' => 'fas fa-plate-wheat',
			'fa-f04b' => 'fas fa-play',
			'fa-f3df' => 'fab fa-playstation',
			'fa-f1e6' => 'fas fa-plug',
			'fa-e55b' => 'fas fa-plug-circle-bolt',
			'fa-e55c' => 'fas fa-plug-circle-check',
			'fa-e55d' => 'fas fa-plug-circle-exclamation',
			'fa-e55e' => 'fas fa-plug-circle-minus',
			'fa-e55f' => 'fas fa-plug-circle-plus',
			'fa-e560' => 'fas fa-plug-circle-xmark',
			'fa-2b' => 'fas fa-plus',
			'fa-e43c' => 'fas fa-plus-minus',
			'fa-f2ce' => 'fas fa-podcast',
			'fa-f2fe' => 'fas fa-poo',
			'fa-f75a' => 'fas fa-poo-storm',
			'fa-f619' => 'fas fa-poop',
			'fa-f011' => 'fas fa-power-off',
			'fa-f5b1' => 'fas fa-prescription',
			'fa-f485' => 'fas fa-prescription-bottle',
			'fa-f486' => 'fas fa-prescription-bottle-medical',
			'fa-f02f' => 'fas fa-print',
			'fa-f288' => 'fab fa-product-hunt',
			'fa-e06a' => 'fas fa-pump-medical',
			'fa-e06b' => 'fas fa-pump-soap',
			'fa-f3e1' => 'fab fa-pushed',
			'fa-f12e' => 'fas fa-puzzle-piece',
			'fa-f3e2' => 'fab fa-python',
			'fa-51' => 'fas fa-q',
			'fa-f1d6' => 'fab fa-qq',
			'fa-f029' => 'fas fa-qrcode',
			'fa-3f' => 'fas fa-question',
			'fa-f459' => 'fab fa-quinscape',
			'fa-f2c4' => 'fab fa-quora',
			'fa-f10d' => 'fas fa-quote-left',
			'fa-f10e' => 'fas fa-quote-right',
			'fa-52' => 'fas fa-r',
			'fa-f4f7' => 'fab fa-r-project',
			'fa-f7b9' => 'fas fa-radiation',
			'fa-f8d7' => 'fas fa-radio',
			'fa-f75b' => 'fas fa-rainbow',
			'fa-e561' => 'fas fa-ranking-star',
			'fa-f7bb' => 'fab fa-raspberry-pi',
			'fa-f2d9' => 'fab fa-ravelry',
			'fa-f41b' => 'fab fa-react',
			'fa-f75d' => 'fab fa-reacteurope',
			'fa-f4d5' => 'fab fa-readme',
			'fa-f1d0' => 'fab fa-rebel',
			'fa-f543' => 'fas fa-receipt',
			'fa-f8d9' => 'fas fa-record-vinyl',
			'fa-f641' => 'fas fa-rectangle-ad',
			'fa-f022' => 'fas fa-rectangle-list',
			'fa-f022' => 'far fa-rectangle-list',
			'fa-f410' => 'fas fa-rectangle-xmark',
			'fa-f410' => 'far fa-rectangle-xmark',
			'fa-f1b8' => 'fas fa-recycle',
			'fa-f3e3' => 'fab fa-red-river',
			'fa-f1a1' => 'fab fa-reddit',
			'fa-f281' => 'fab fa-reddit-alien',
			'fa-f7bc' => 'fab fa-redhat',
			'fa-f25d' => 'fas fa-registered',
			'fa-f25d' => 'far fa-registered',
			'fa-f18b' => 'fab fa-renren',
			'fa-f363' => 'fas fa-repeat',
			'fa-f3e5' => 'fas fa-reply',
			'fa-f122' => 'fas fa-reply-all',
			'fa-f3e6' => 'fab fa-replyd',
			'fa-f75e' => 'fas fa-republican',
			'fa-f4f8' => 'fab fa-researchgate',
			'fa-f3e7' => 'fab fa-resolving',
			'fa-f7bd' => 'fas fa-restroom',
			'fa-f079' => 'fas fa-retweet',
			'fa-f5b2' => 'fab fa-rev',
			'fa-f4d6' => 'fas fa-ribbon',
			'fa-f2f5' => 'fas fa-right-from-bracket',
			'fa-f362' => 'fas fa-right-left',
			'fa-f30b' => 'fas fa-right-long',
			'fa-f2f6' => 'fas fa-right-to-bracket',
			'fa-f70b' => 'fas fa-ring',
			'fa-f018' => 'fas fa-road',
			'fa-e562' => 'fas fa-road-barrier',
			'fa-e563' => 'fas fa-road-bridge',
			'fa-e564' => 'fas fa-road-circle-check',
			'fa-e565' => 'fas fa-road-circle-exclamation',
			'fa-e566' => 'fas fa-road-circle-xmark',
			'fa-e567' => 'fas fa-road-lock',
			'fa-e568' => 'fas fa-road-spikes',
			'fa-f544' => 'fas fa-robot',
			'fa-f135' => 'fas fa-rocket',
			'fa-f3e8' => 'fab fa-rocketchat',
			'fa-f3e9' => 'fab fa-rockrms',
			'fa-f2f1' => 'fas fa-rotate',
			'fa-f2ea' => 'fas fa-rotate-left',
			'fa-f2f9' => 'fas fa-rotate-right',
			'fa-f4d7' => 'fas fa-route',
			'fa-f09e' => 'fas fa-rss',
			'fa-f158' => 'fas fa-ruble-sign',
			'fa-e569' => 'fas fa-rug',
			'fa-f545' => 'fas fa-ruler',
			'fa-f546' => 'fas fa-ruler-combined',
			'fa-f547' => 'fas fa-ruler-horizontal',
			'fa-f548' => 'fas fa-ruler-vertical',
			'fa-f156' => 'fas fa-rupee-sign',
			'fa-e23d' => 'fas fa-rupiah-sign',
			'fa-e07a' => 'fab fa-rust',
			'fa-53' => 'fas fa-s',
			'fa-f81d' => 'fas fa-sack-dollar',
			'fa-e56a' => 'fas fa-sack-xmark',
			'fa-f267' => 'fab fa-safari',
			'fa-e445' => 'fas fa-sailboat',
			'fa-f83b' => 'fab fa-salesforce',
			'fa-f41e' => 'fab fa-sass',
			'fa-f7bf' => 'fas fa-satellite',
			'fa-f7c0' => 'fas fa-satellite-dish',
			'fa-f24e' => 'fas fa-scale-balanced',
			'fa-f515' => 'fas fa-scale-unbalanced',
			'fa-f516' => 'fas fa-scale-unbalanced-flip',
			'fa-f3ea' => 'fab fa-schlix',
			'fa-f549' => 'fas fa-school',
			'fa-e56b' => 'fas fa-school-circle-check',
			'fa-e56c' => 'fas fa-school-circle-exclamation',
			'fa-e56d' => 'fas fa-school-circle-xmark',
			'fa-e56e' => 'fas fa-school-flag',
			'fa-e56f' => 'fas fa-school-lock',
			'fa-f0c4' => 'fas fa-scissors',
			'fa-e570' => 'fab fa-screenpal',
			'fa-f54a' => 'fas fa-screwdriver',
			'fa-f7d9' => 'fas fa-screwdriver-wrench',
			'fa-f28a' => 'fab fa-scribd',
			'fa-f70e' => 'fas fa-scroll',
			'fa-f6a0' => 'fas fa-scroll-torah',
			'fa-f7c2' => 'fas fa-sd-card',
			'fa-f3eb' => 'fab fa-searchengin',
			'fa-e447' => 'fas fa-section',
			'fa-f4d8' => 'fas fa-seedling',
			'fa-f2da' => 'fab fa-sellcast',
			'fa-f213' => 'fab fa-sellsy',
			'fa-f233' => 'fas fa-server',
			'fa-f3ec' => 'fab fa-servicestack',
			'fa-f61f' => 'fas fa-shapes',
			'fa-f064' => 'fas fa-share',
			'fa-f14d' => 'fas fa-share-from-square',
			'fa-f14d' => 'far fa-share-from-square',
			'fa-f1e0' => 'fas fa-share-nodes',
			'fa-e571' => 'fas fa-sheet-plastic',
			'fa-f20b' => 'fas fa-shekel-sign',
			'fa-f132' => 'fas fa-shield',
			'fa-e572' => 'fas fa-shield-cat',
			'fa-e573' => 'fas fa-shield-dog',
			'fa-f3ed' => 'fas fa-shield-halved',
			'fa-e574' => 'fas fa-shield-heart',
			'fa-e06c' => 'fas fa-shield-virus',
			'fa-f21a' => 'fas fa-ship',
			'fa-f553' => 'fas fa-shirt',
			'fa-f214' => 'fab fa-shirtsinbulk',
			'fa-f54b' => 'fas fa-shoe-prints',
			'fa-f54f' => 'fas fa-shop',
			'fa-e4a5' => 'fas fa-shop-lock',
			'fa-e070' => 'fas fa-shop-slash',
			'fa-e057' => 'fab fa-shopify',
			'fa-f5b5' => 'fab fa-shopware',
			'fa-f2cc' => 'fas fa-shower',
			'fa-e448' => 'fas fa-shrimp',
			'fa-f074' => 'fas fa-shuffle',
			'fa-f197' => 'fas fa-shuttle-space',
			'fa-f4d9' => 'fas fa-sign-hanging',
			'fa-f012' => 'fas fa-signal',
			'fa-f5b7' => 'fas fa-signature',
			'fa-f277' => 'fas fa-signs-post',
			'fa-f7c4' => 'fas fa-sim-card',
			'fa-f215' => 'fab fa-simplybuilt',
			'fa-e06d' => 'fas fa-sink',
			'fa-f3ee' => 'fab fa-sistrix',
			'fa-f0e8' => 'fas fa-sitemap',
			'fa-f512' => 'fab fa-sith',
			'fa-e44a' => 'fab fa-sitrox',
			'fa-f7c6' => 'fab fa-sketch',
			'fa-f54c' => 'fas fa-skull',
			'fa-f714' => 'fas fa-skull-crossbones',
			'fa-f216' => 'fab fa-skyatlas',
			'fa-f17e' => 'fab fa-skype',
			'fa-f198' => 'fab fa-slack',
			'fa-f715' => 'fas fa-slash',
			'fa-f7cc' => 'fas fa-sleigh',
			'fa-f1de' => 'fas fa-sliders',
			'fa-f1e7' => 'fab fa-slideshare',
			'fa-f75f' => 'fas fa-smog',
			'fa-f48d' => 'fas fa-smoking',
			'fa-f2ab' => 'fab fa-snapchat',
			'fa-f2dc' => 'fas fa-snowflake',
			'fa-f2dc' => 'far fa-snowflake',
			'fa-f7d0' => 'fas fa-snowman',
			'fa-f7d2' => 'fas fa-snowplow',
			'fa-e06e' => 'fas fa-soap',
			'fa-f696' => 'fas fa-socks',
			'fa-f5ba' => 'fas fa-solar-panel',
			'fa-f0dc' => 'fas fa-sort',
			'fa-f0dd' => 'fas fa-sort-down',
			'fa-f0de' => 'fas fa-sort-up',
			'fa-f1be' => 'fab fa-soundcloud',
			'fa-f7d3' => 'fab fa-sourcetree',
			'fa-f5bb' => 'fas fa-spa',
			'fa-e5ac' => 'fab fa-space-awesome',
			'fa-f67b' => 'fas fa-spaghetti-monster-flying',
			'fa-f3f3' => 'fab fa-speakap',
			'fa-f83c' => 'fab fa-speaker-deck',
			'fa-f891' => 'fas fa-spell-check',
			'fa-f717' => 'fas fa-spider',
			'fa-f110' => 'fas fa-spinner',
			'fa-f5bc' => 'fas fa-splotch',
			'fa-f2e5' => 'fas fa-spoon',
			'fa-f1bc' => 'fab fa-spotify',
			'fa-f5bd' => 'fas fa-spray-can',
			'fa-f5d0' => 'fas fa-spray-can-sparkles',
			'fa-f0c8' => 'fas fa-square',
			'fa-f0c8' => 'far fa-square',
			'fa-f14c' => 'fas fa-square-arrow-up-right',
			'fa-f1b5' => 'fab fa-square-behance',
			'fa-f150' => 'fas fa-square-caret-down',
			'fa-f150' => 'far fa-square-caret-down',
			'fa-f191' => 'fas fa-square-caret-left',
			'fa-f191' => 'far fa-square-caret-left',
			'fa-f152' => 'fas fa-square-caret-right',
			'fa-f152' => 'far fa-square-caret-right',
			'fa-f151' => 'fas fa-square-caret-up',
			'fa-f151' => 'far fa-square-caret-up',
			'fa-f14a' => 'fas fa-square-check',
			'fa-f14a' => 'far fa-square-check',
			'fa-f397' => 'fab fa-square-dribbble',
			'fa-f199' => 'fas fa-square-envelope',
			'fa-f082' => 'fab fa-square-facebook',
			'fa-e5ad' => 'fab fa-square-font-awesome',
			'fa-f35c' => 'fab fa-square-font-awesome-stroke',
			'fa-f45c' => 'fas fa-square-full',
			'fa-f45c' => 'far fa-square-full',
			'fa-f1d2' => 'fab fa-square-git',
			'fa-f092' => 'fab fa-square-github',
			'fa-e5ae' => 'fab fa-square-gitlab',
			'fa-f0d4' => 'fab fa-square-google-plus',
			'fa-f0fd' => 'fas fa-square-h',
			'fa-f3af' => 'fab fa-square-hacker-news',
			'fa-e055' => 'fab fa-square-instagram',
			'fa-f3b9' => 'fab fa-square-js',
			'fa-f203' => 'fab fa-square-lastfm',
			'fa-f146' => 'fas fa-square-minus',
			'fa-f146' => 'far fa-square-minus',
			'fa-e576' => 'fas fa-square-nfi',
			'fa-f264' => 'fab fa-square-odnoklassniki',
			'fa-f540' => 'fas fa-square-parking',
			'fa-f14b' => 'fas fa-square-pen',
			'fa-e577' => 'fas fa-square-person-confined',
			'fa-f098' => 'fas fa-square-phone',
			'fa-f87b' => 'fas fa-square-phone-flip',
			'fa-e01e' => 'fab fa-square-pied-piper',
			'fa-f0d3' => 'fab fa-square-pinterest',
			'fa-f0fe' => 'fas fa-square-plus',
			'fa-f0fe' => 'far fa-square-plus',
			'fa-f682' => 'fas fa-square-poll-horizontal',
			'fa-f681' => 'fas fa-square-poll-vertical',
			'fa-f1a2' => 'fab fa-square-reddit',
			'fa-f698' => 'fas fa-square-root-variable',
			'fa-f143' => 'fas fa-square-rss',
			'fa-f1e1' => 'fas fa-square-share-nodes',
			'fa-f2ad' => 'fab fa-square-snapchat',
			'fa-f1b7' => 'fab fa-square-steam',
			'fa-e619' => 'fab fa-square-threads',
			'fa-f174' => 'fab fa-square-tumblr',
			'fa-f081' => 'fab fa-square-twitter',
			'fa-f360' => 'fas fa-square-up-right',
			'fa-f2aa' => 'fab fa-square-viadeo',
			'fa-f194' => 'fab fa-square-vimeo',
			'fa-e578' => 'fas fa-square-virus',
			'fa-f40c' => 'fab fa-square-whatsapp',
			'fa-e61a' => 'fab fa-square-x-twitter',
			'fa-f169' => 'fab fa-square-xing',
			'fa-f2d3' => 'fas fa-square-xmark',
			'fa-f431' => 'fab fa-square-youtube',
			'fa-f5be' => 'fab fa-squarespace',
			'fa-f18d' => 'fab fa-stack-exchange',
			'fa-f16c' => 'fab fa-stack-overflow',
			'fa-f842' => 'fab fa-stackpath',
			'fa-e579' => 'fas fa-staff-snake',
			'fa-e289' => 'fas fa-stairs',
			'fa-f5bf' => 'fas fa-stamp',
			'fa-e5af' => 'fas fa-stapler',
			'fa-f005' => 'fas fa-star',
			'fa-f005' => 'far fa-star',
			'fa-f699' => 'fas fa-star-and-crescent',
			'fa-f089' => 'fas fa-star-half',
			'fa-f089' => 'far fa-star-half',
			'fa-f5c0' => 'fas fa-star-half-stroke',
			'fa-f5c0' => 'far fa-star-half-stroke',
			'fa-f69a' => 'fas fa-star-of-david',
			'fa-f621' => 'fas fa-star-of-life',
			'fa-f3f5' => 'fab fa-staylinked',
			'fa-f1b6' => 'fab fa-steam',
			'fa-f3f6' => 'fab fa-steam-symbol',
			'fa-f154' => 'fas fa-sterling-sign',
			'fa-f0f1' => 'fas fa-stethoscope',
			'fa-f3f7' => 'fab fa-sticker-mule',
			'fa-f04d' => 'fas fa-stop',
			'fa-f2f2' => 'fas fa-stopwatch',
			'fa-e06f' => 'fas fa-stopwatch-20',
			'fa-f54e' => 'fas fa-store',
			'fa-e071' => 'fas fa-store-slash',
			'fa-f428' => 'fab fa-strava',
			'fa-f21d' => 'fas fa-street-view',
			'fa-f0cc' => 'fas fa-strikethrough',
			'fa-f429' => 'fab fa-stripe',
			'fa-f42a' => 'fab fa-stripe-s',
			'fa-f551' => 'fas fa-stroopwafel',
			'fa-e5c7' => 'fab fa-stubber',
			'fa-f3f8' => 'fab fa-studiovinari',
			'fa-f1a4' => 'fab fa-stumbleupon',
			'fa-f1a3' => 'fab fa-stumbleupon-circle',
			'fa-f12c' => 'fas fa-subscript',
			'fa-f0f2' => 'fas fa-suitcase',
			'fa-f0fa' => 'fas fa-suitcase-medical',
			'fa-f5c1' => 'fas fa-suitcase-rolling',
			'fa-f185' => 'fas fa-sun',
			'fa-f185' => 'far fa-sun',
			'fa-e57a' => 'fas fa-sun-plant-wilt',
			'fa-f2dd' => 'fab fa-superpowers',
			'fa-f12b' => 'fas fa-superscript',
			'fa-f3f9' => 'fab fa-supple',
			'fa-f7d6' => 'fab fa-suse',
			'fa-f5c3' => 'fas fa-swatchbook',
			'fa-f8e1' => 'fab fa-swift',
			'fa-f83d' => 'fab fa-symfony',
			'fa-f69b' => 'fas fa-synagogue',
			'fa-f48e' => 'fas fa-syringe',
			'fa-54' => 'fas fa-t',
			'fa-f0ce' => 'fas fa-table',
			'fa-f00a' => 'fas fa-table-cells',
			'fa-f009' => 'fas fa-table-cells-large',
			'fa-f0db' => 'fas fa-table-columns',
			'fa-f00b' => 'fas fa-table-list',
			'fa-f45d' => 'fas fa-table-tennis-paddle-ball',
			'fa-f3fb' => 'fas fa-tablet',
			'fa-f10a' => 'fas fa-tablet-button',
			'fa-f3fa' => 'fas fa-tablet-screen-button',
			'fa-f490' => 'fas fa-tablets',
			'fa-f566' => 'fas fa-tachograph-digital',
			'fa-f02b' => 'fas fa-tag',
			'fa-f02c' => 'fas fa-tags',
			'fa-f4db' => 'fas fa-tape',
			'fa-e57b' => 'fas fa-tarp',
			'fa-e57c' => 'fas fa-tarp-droplet',
			'fa-f1ba' => 'fas fa-taxi',
			'fa-f4f9' => 'fab fa-teamspeak',
			'fa-f62e' => 'fas fa-teeth',
			'fa-f62f' => 'fas fa-teeth-open',
			'fa-f2c6' => 'fab fa-telegram',
			'fa-e03f' => 'fas fa-temperature-arrow-down',
			'fa-e040' => 'fas fa-temperature-arrow-up',
			'fa-f2cb' => 'fas fa-temperature-empty',
			'fa-f2c7' => 'fas fa-temperature-full',
			'fa-f2c9' => 'fas fa-temperature-half',
			'fa-f769' => 'fas fa-temperature-high',
			'fa-f76b' => 'fas fa-temperature-low',
			'fa-f2ca' => 'fas fa-temperature-quarter',
			'fa-f2c8' => 'fas fa-temperature-three-quarters',
			'fa-f1d5' => 'fab fa-tencent-weibo',
			'fa-f7d7' => 'fas fa-tenge-sign',
			'fa-e57d' => 'fas fa-tent',
			'fa-e57e' => 'fas fa-tent-arrow-down-to-line',
			'fa-e57f' => 'fas fa-tent-arrow-left-right',
			'fa-e580' => 'fas fa-tent-arrow-turn-left',
			'fa-e581' => 'fas fa-tent-arrows-down',
			'fa-e582' => 'fas fa-tents',
			'fa-f120' => 'fas fa-terminal',
			'fa-f034' => 'fas fa-text-height',
			'fa-f87d' => 'fas fa-text-slash',
			'fa-f035' => 'fas fa-text-width',
			'fa-f69d' => 'fab fa-the-red-yeti',
			'fa-f5c6' => 'fab fa-themeco',
			'fa-f2b2' => 'fab fa-themeisle',
			'fa-f491' => 'fas fa-thermometer',
			'fa-f731' => 'fab fa-think-peaks',
			'fa-e618' => 'fab fa-threads',
			'fa-f165' => 'fas fa-thumbs-down',
			'fa-f165' => 'far fa-thumbs-down',
			'fa-f164' => 'fas fa-thumbs-up',
			'fa-f164' => 'far fa-thumbs-up',
			'fa-f08d' => 'fas fa-thumbtack',
			'fa-f145' => 'fas fa-ticket',
			'fa-f3ff' => 'fas fa-ticket-simple',
			'fa-e07b' => 'fab fa-tiktok',
			'fa-e29c' => 'fas fa-timeline',
			'fa-f204' => 'fas fa-toggle-off',
			'fa-f205' => 'fas fa-toggle-on',
			'fa-f7d8' => 'fas fa-toilet',
			'fa-f71e' => 'fas fa-toilet-paper',
			'fa-e072' => 'fas fa-toilet-paper-slash',
			'fa-e583' => 'fas fa-toilet-portable',
			'fa-e584' => 'fas fa-toilets-portable',
			'fa-f552' => 'fas fa-toolbox',
			'fa-f5c9' => 'fas fa-tooth',
			'fa-f6a1' => 'fas fa-torii-gate',
			'fa-f76f' => 'fas fa-tornado',
			'fa-f519' => 'fas fa-tower-broadcast',
			'fa-e585' => 'fas fa-tower-cell',
			'fa-e586' => 'fas fa-tower-observation',
			'fa-f722' => 'fas fa-tractor',
			'fa-f513' => 'fab fa-trade-federation',
			'fa-f25c' => 'fas fa-trademark',
			'fa-f637' => 'fas fa-traffic-light',
			'fa-e041' => 'fas fa-trailer',
			'fa-f238' => 'fas fa-train',
			'fa-f239' => 'fas fa-train-subway',
			'fa-e5b4' => 'fas fa-train-tram',
			'fa-f225' => 'fas fa-transgender',
			'fa-f1f8' => 'fas fa-trash',
			'fa-f829' => 'fas fa-trash-arrow-up',
			'fa-f2ed' => 'fas fa-trash-can',
			'fa-f2ed' => 'far fa-trash-can',
			'fa-f82a' => 'fas fa-trash-can-arrow-up',
			'fa-f1bb' => 'fas fa-tree',
			'fa-e587' => 'fas fa-tree-city',
			'fa-f181' => 'fab fa-trello',
			'fa-f071' => 'fas fa-triangle-exclamation',
			'fa-f091' => 'fas fa-trophy',
			'fa-e589' => 'fas fa-trowel',
			'fa-e58a' => 'fas fa-trowel-bricks',
			'fa-f0d1' => 'fas fa-truck',
			'fa-e58b' => 'fas fa-truck-arrow-right',
			'fa-e58c' => 'fas fa-truck-droplet',
			'fa-f48b' => 'fas fa-truck-fast',
			'fa-e58d' => 'fas fa-truck-field',
			'fa-e58e' => 'fas fa-truck-field-un',
			'fa-e2b7' => 'fas fa-truck-front',
			'fa-f0f9' => 'fas fa-truck-medical',
			'fa-f63b' => 'fas fa-truck-monster',
			'fa-f4df' => 'fas fa-truck-moving',
			'fa-f63c' => 'fas fa-truck-pickup',
			'fa-e58f' => 'fas fa-truck-plane',
			'fa-f4de' => 'fas fa-truck-ramp-box',
			'fa-f1e4' => 'fas fa-tty',
			'fa-f173' => 'fab fa-tumblr',
			'fa-e2bb' => 'fas fa-turkish-lira-sign',
			'fa-f3be' => 'fas fa-turn-down',
			'fa-f3bf' => 'fas fa-turn-up',
			'fa-f26c' => 'fas fa-tv',
			'fa-f1e8' => 'fab fa-twitch',
			'fa-f099' => 'fab fa-twitter',
			'fa-f42b' => 'fab fa-typo3',
			'fa-55' => 'fas fa-u',
			'fa-f402' => 'fab fa-uber',
			'fa-f7df' => 'fab fa-ubuntu',
			'fa-f403' => 'fab fa-uikit',
			'fa-f8e8' => 'fab fa-umbraco',
			'fa-f0e9' => 'fas fa-umbrella',
			'fa-f5ca' => 'fas fa-umbrella-beach',
			'fa-e084' => 'fab fa-uncharted',
			'fa-f0cd' => 'fas fa-underline',
			'fa-f404' => 'fab fa-uniregistry',
			'fa-e049' => 'fab fa-unity',
			'fa-f29a' => 'fas fa-universal-access',
			'fa-f09c' => 'fas fa-unlock',
			'fa-f13e' => 'fas fa-unlock-keyhole',
			'fa-e07c' => 'fab fa-unsplash',
			'fa-f405' => 'fab fa-untappd',
			'fa-f338' => 'fas fa-up-down',
			'fa-f0b2' => 'fas fa-up-down-left-right',
			'fa-f30c' => 'fas fa-up-long',
			'fa-f424' => 'fas fa-up-right-and-down-left-from-center',
			'fa-f35d' => 'fas fa-up-right-from-square',
			'fa-f093' => 'fas fa-upload',
			'fa-f7e0' => 'fab fa-ups',
			'fa-f287' => 'fab fa-usb',
			'fa-f007' => 'fas fa-user',
			'fa-f007' => 'far fa-user',
			'fa-f4fb' => 'fas fa-user-astronaut',
			'fa-f4fc' => 'fas fa-user-check',
			'fa-f4fd' => 'fas fa-user-clock',
			'fa-f0f0' => 'fas fa-user-doctor',
			'fa-f4fe' => 'fas fa-user-gear',
			'fa-f501' => 'fas fa-user-graduate',
			'fa-f500' => 'fas fa-user-group',
			'fa-f728' => 'fas fa-user-injured',
			'fa-f406' => 'fas fa-user-large',
			'fa-f4fa' => 'fas fa-user-large-slash',
			'fa-f502' => 'fas fa-user-lock',
			'fa-f503' => 'fas fa-user-minus',
			'fa-f504' => 'fas fa-user-ninja',
			'fa-f82f' => 'fas fa-user-nurse',
			'fa-f4ff' => 'fas fa-user-pen',
			'fa-f234' => 'fas fa-user-plus',
			'fa-f21b' => 'fas fa-user-secret',
			'fa-f505' => 'fas fa-user-shield',
			'fa-f506' => 'fas fa-user-slash',
			'fa-f507' => 'fas fa-user-tag',
			'fa-f508' => 'fas fa-user-tie',
			'fa-f235' => 'fas fa-user-xmark',
			'fa-f0c0' => 'fas fa-users',
			'fa-e591' => 'fas fa-users-between-lines',
			'fa-f509' => 'fas fa-users-gear',
			'fa-e592' => 'fas fa-users-line',
			'fa-e593' => 'fas fa-users-rays',
			'fa-e594' => 'fas fa-users-rectangle',
			'fa-e073' => 'fas fa-users-slash',
			'fa-e595' => 'fas fa-users-viewfinder',
			'fa-f7e1' => 'fab fa-usps',
			'fa-f407' => 'fab fa-ussunnah',
			'fa-f2e7' => 'fas fa-utensils',
			'fa-56' => 'fas fa-v',
			'fa-f408' => 'fab fa-vaadin',
			'fa-f5b6' => 'fas fa-van-shuttle',
			'fa-e2c5' => 'fas fa-vault',
			'fa-f5cb' => 'fas fa-vector-square',
			'fa-f221' => 'fas fa-venus',
			'fa-f226' => 'fas fa-venus-double',
			'fa-f228' => 'fas fa-venus-mars',
			'fa-e085' => 'fas fa-vest',
			'fa-e086' => 'fas fa-vest-patches',
			'fa-f237' => 'fab fa-viacoin',
			'fa-f2a9' => 'fab fa-viadeo',
			'fa-f492' => 'fas fa-vial',
			'fa-e596' => 'fas fa-vial-circle-check',
			'fa-e597' => 'fas fa-vial-virus',
			'fa-f493' => 'fas fa-vials',
			'fa-f409' => 'fab fa-viber',
			'fa-f03d' => 'fas fa-video',
			'fa-f4e2' => 'fas fa-video-slash',
			'fa-f6a7' => 'fas fa-vihara',
			'fa-f40a' => 'fab fa-vimeo',
			'fa-f27d' => 'fab fa-vimeo-v',
			'fa-f1ca' => 'fab fa-vine',
			'fa-e074' => 'fas fa-virus',
			'fa-e4a8' => 'fas fa-virus-covid',
			'fa-e4a9' => 'fas fa-virus-covid-slash',
			'fa-e075' => 'fas fa-virus-slash',
			'fa-e076' => 'fas fa-viruses',
			'fa-f189' => 'fab fa-vk',
			'fa-f40b' => 'fab fa-vnv',
			'fa-f897' => 'fas fa-voicemail',
			'fa-f770' => 'fas fa-volcano',
			'fa-f45f' => 'fas fa-volleyball',
			'fa-f028' => 'fas fa-volume-high',
			'fa-f027' => 'fas fa-volume-low',
			'fa-f026' => 'fas fa-volume-off',
			'fa-f6a9' => 'fas fa-volume-xmark',
			'fa-f729' => 'fas fa-vr-cardboard',
			'fa-f41f' => 'fab fa-vuejs',
			'fa-57' => 'fas fa-w',
			'fa-f8ef' => 'fas fa-walkie-talkie',
			'fa-f555' => 'fas fa-wallet',
			'fa-f0d0' => 'fas fa-wand-magic',
			'fa-e2ca' => 'fas fa-wand-magic-sparkles',
			'fa-f72b' => 'fas fa-wand-sparkles',
			'fa-f494' => 'fas fa-warehouse',
			'fa-e087' => 'fab fa-watchman-monitoring',
			'fa-f773' => 'fas fa-water',
			'fa-f5c5' => 'fas fa-water-ladder',
			'fa-f83e' => 'fas fa-wave-square',
			'fa-f83f' => 'fab fa-waze',
			'fa-f5cc' => 'fab fa-weebly',
			'fa-f18a' => 'fab fa-weibo',
			'fa-f5cd' => 'fas fa-weight-hanging',
			'fa-f496' => 'fas fa-weight-scale',
			'fa-f1d7' => 'fab fa-weixin',
			'fa-f232' => 'fab fa-whatsapp',
			'fa-e2cd' => 'fas fa-wheat-awn',
			'fa-e598' => 'fas fa-wheat-awn-circle-exclamation',
			'fa-f193' => 'fas fa-wheelchair',
			'fa-e2ce' => 'fas fa-wheelchair-move',
			'fa-f7a0' => 'fas fa-whiskey-glass',
			'fa-f40d' => 'fab fa-whmcs',
			'fa-f1eb' => 'fas fa-wifi',
			'fa-f266' => 'fab fa-wikipedia-w',
			'fa-f72e' => 'fas fa-wind',
			'fa-f2d0' => 'fas fa-window-maximize',
			'fa-f2d0' => 'far fa-window-maximize',
			'fa-f2d1' => 'fas fa-window-minimize',
			'fa-f2d1' => 'far fa-window-minimize',
			'fa-f2d2' => 'fas fa-window-restore',
			'fa-f2d2' => 'far fa-window-restore',
			'fa-f17a' => 'fab fa-windows',
			'fa-f72f' => 'fas fa-wine-bottle',
			'fa-f4e3' => 'fas fa-wine-glass',
			'fa-f5ce' => 'fas fa-wine-glass-empty',
			'fa-e2d0' => 'fab fa-wirsindhandwerk',
			'fa-f5cf' => 'fab fa-wix',
			'fa-f730' => 'fab fa-wizards-of-the-coast',
			'fa-e088' => 'fab fa-wodu',
			'fa-f514' => 'fab fa-wolf-pack-battalion',
			'fa-f159' => 'fas fa-won-sign',
			'fa-f19a' => 'fab fa-wordpress',
			'fa-f411' => 'fab fa-wordpress-simple',
			'fa-e599' => 'fas fa-worm',
			'fa-f297' => 'fab fa-wpbeginner',
			'fa-f2de' => 'fab fa-wpexplorer',
			'fa-f298' => 'fab fa-wpforms',
			'fa-f3e4' => 'fab fa-wpressr',
			'fa-f0ad' => 'fas fa-wrench',
			'fa-58' => 'fas fa-x',
			'fa-f497' => 'fas fa-x-ray',
			'fa-e61b' => 'fab fa-x-twitter',
			'fa-f412' => 'fab fa-xbox',
			'fa-f168' => 'fab fa-xing',
			'fa-f00d' => 'fas fa-xmark',
			'fa-e59a' => 'fas fa-xmarks-lines',
			'fa-59' => 'fas fa-y',
			'fa-f23b' => 'fab fa-y-combinator',
			'fa-f19e' => 'fab fa-yahoo',
			'fa-f840' => 'fab fa-yammer',
			'fa-f413' => 'fab fa-yandex',
			'fa-f414' => 'fab fa-yandex-international',
			'fa-f7e3' => 'fab fa-yarn',
			'fa-f1e9' => 'fab fa-yelp',
			'fa-f157' => 'fas fa-yen-sign',
			'fa-f6ad' => 'fas fa-yin-yang',
			'fa-f2b1' => 'fab fa-yoast',
			'fa-f167' => 'fab fa-youtube',
			'fa-5a' => 'fas fa-z',
			'fa-f63f' => 'fab fa-zhihu',
		);

		$icons = apply_filters( "megamenu_fontawesome_6_icons", $icons );

		return $icons;

	}

	/**
	public function fa_icons () {
	    $content = $this->file_get_contents_curl('https://raw.githubusercontent.com/FortAwesome/Font-Awesome/6.x/metadata/icons.json');
	    $json = json_decode($content);
	    $icons = [];

	    foreach ($json as $icon => $value) {
	        foreach ($value->styles as $style) {
	        	$unicode = $value->unicode;
	        	$style = "fa" . substr($style, 0 ,1);
	            echo "'fa-{$unicode}' => '{$style} fa-{$icon}'," . "<br />";
	        }
	    }
	}

	public function file_get_contents_curl( $url ) {

	  $ch = curl_init();

	  curl_setopt( $ch, CURLOPT_AUTOREFERER, TRUE );
	  curl_setopt( $ch, CURLOPT_HEADER, 0 );
	  curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
	  curl_setopt( $ch, CURLOPT_URL, $url );
	  curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, TRUE );

	  $data = curl_exec( $ch );
	  curl_close( $ch );

	  return $data;

	}
	**/
}

endif;