<?php

/*
 * Plugin Name: Max Mega Menu - Pro Addon
 * Plugin URI:  https://www.megamenu.com
 * Description: Extends the free version of Max Mega Menu with additional functionality.
 * Version:     2.4.3
 * Author:      megamenu.com
 * Author URI:  https://www.megamenu.com
 * Copyright:   2020 Tom Hemsley (https://www.megamenu.com)
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // disable direct access
}

if ( ! class_exists('Mega_Menu_Pro') ) :

/**
 *
 */
class Mega_Menu_Pro {

	/**
	 * @var string
	 */
	public $version = '2.4.3';


	/**
	 * Init
	 *
	 * @since 1.0
	 */
	public static function init() {
		$plugin = new self();
	}


	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	public function __construct() {

		define( "MEGAMENU_PRO_VERSION", $this->version );
		define( "MEGAMENU_PRO_PLUGIN_FILE", __FILE__ );

		add_action( "init", array( $this, 'load_plugin_textdomain' ) );
		add_action( "widgets_init", array( $this, 'register_widget' ) );
		add_filter( "megamenu_versions", array( $this, 'add_version_to_header' ) );
		add_action( "admin_init", array( $this, 'install_upgrade_check' ) );
		add_action( "megamenu_enqueue_admin_scripts", array( $this, 'enqueue_nav_menu_scripts' ) );
		add_action( "megamenu_admin_scripts", array( $this, 'enqueue_admin_scripts') );
		add_filter( "megamenu_nav_menu_objects_after", array( $this, 'apply_classes_to_menu_items' ), 7, 2 );
		add_action( "admin_notices", array( $this, 'check_megamenu_is_installed' ) );
		add_action( "megamenu_enqueue_scripts", array( $this, 'enqueue_public_scripts' ), 999 );
		add_action( "wp_enqueue_scripts", array( $this, 'enqueue_public_scripts' ), 999 );
		add_action( "megamenu_general_settings", array( $this, 'add_icons_settings_to_general_settings' ), 21 );
		add_filter( "megamenu_submitted_settings", array( $this, 'populate_empty_checkbox_values'), 9 );
		add_action( "set_transient_megamenu_css", array( $this, 'set_megamenu_pro_css_version' ), 10, 3);

		$this->load();
	}

	/**
	 * Initialise translations
	 *
	 * @since 1.0
	 */
	public function load_plugin_textdomain() {
		load_plugin_textdomain( 'megamenu-pro', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * Detect new or updated installations and run actions accordingly.
	 *
	 * @since 1.8
	 */
	public function install_upgrade_check() {
		$version = get_option( "megamenu_pro_version" );

		if ( $version ) {
			if ( version_compare( $this->version, $version, '!=' ) ) {
				update_option( "megamenu_pro_version", $this->version );
				do_action( "megamenu_pro_after_update" );
			}

		} else {
			add_option( "megamenu_pro_version", $this->version );
			add_option( "megamenu_pro_initial_version", $this->version );
			do_action( "megamenu_pro_after_install" );
		}
	}


	/**
	 * Record the version that the CSS was generated against
	 * @param mixed  $value      Transient value.
	 * @param int    $expiration Time until expiration in seconds.
	 * @param string $transient  The name of the transient.
	 * @since 2.1: Start saving this as an option as memcached can return incorrect values
	 */
	public function set_megamenu_pro_css_version( $value, $expiration, $transient ) {
		update_option( 'megamenu_pro_css_version', MEGAMENU_PRO_VERSION );
		delete_transient( 'megamenu_pro_css_version' ); // clean up transient from old versions
	}


	/**
	 * Populate empty checkbox values when the general settings form is submitted
	 *
	 * @since 1.8
	 * @param array $settings
	 */
	public function populate_empty_checkbox_values( $settings ) {

		if ( ! isset( $settings['enqueue_fa_4'] ) ) {
			$settings['enqueue_fa_4'] = 'disabled';
		}

		if ( ! isset( $settings['enqueue_fa_5'] ) ) {
			$settings['enqueue_fa_5'] = 'disabled';
		}

		if ( ! isset( $settings['enqueue_fa_6'] ) ) {
			$settings['enqueue_fa_6'] = 'disabled';
		}

		if ( ! isset( $settings['enqueue_genericons'] ) ) {
			$settings['enqueue_genericons'] = 'disabled';
		}

		return $settings;

	}

	/**
	 * Add the font awesome version selector to General Settings page
	 *
	 * @since 1.8
	 * @param array $settings
	 */
	public function add_icons_settings_to_general_settings( $saved_settings ) {
		$enqueue_fa_4 = isset( $saved_settings['enqueue_fa_4'] ) ? $saved_settings['enqueue_fa_4'] : 'enabled';
		$enqueue_fa_5 = isset( $saved_settings['enqueue_fa_5'] ) ? $saved_settings['enqueue_fa_5'] : 'enabled';
		$enqueue_fa_6 = isset( $saved_settings['enqueue_fa_6'] ) ? $saved_settings['enqueue_fa_6'] : 'enabled';
		$enqueue_genericons = isset( $saved_settings['enqueue_genericons'] ) ? $saved_settings['enqueue_genericons'] : 'enabled';

		?>

		<h3><?php _e("Icons", "megamenu-pro"); ?></h3>

		<table>
			<tr>
				<td class='mega-name'>
					<?php _e("Enqueue Icon Styles", "megamenu-pro"); ?>
					<div class='mega-description'>
						<?php _e("", "megamenu-pro"); ?>
					</div>
				</td>
				<td class='mega-value'>
					<table class='mega-inner-table'>
						<?php if ($this->font_awesome_4_is_enabled() ): ?>
							<tr>
								<td>
									<input type="checkbox" name="settings[enqueue_fa_4]" value="enabled" <?php checked( $enqueue_fa_4, 'enabled' ); ?> />
								</td>
								<td>
									<?php _e("Font Awesome 4", "megamenu-pro"); ?>
								</td>
							</tr>
						<?php endif; ?>
						<?php if ($this->font_awesome_5_is_enabled() ): ?>
							<tr>
								<td>
									<input type="checkbox" name="settings[enqueue_fa_5]" value="enabled" <?php checked( $enqueue_fa_5, 'enabled' ); ?> />
								</td>
								<td>
									<?php _e("Font Awesome 5", "megamenu-pro"); ?>
								</td>
							</tr>
						<?php endif; ?>
						<tr>
							<td>
								<input type="checkbox" name="settings[enqueue_fa_6]" value="enabled" <?php checked( $enqueue_fa_6, 'enabled' ); ?> />
							</td>
							<td>
								<?php _e("Font Awesome 6", "megamenu-pro"); ?>
							</td>
						</tr>
						<tr>
							<td>
								<input type="checkbox" name="settings[enqueue_genericons]" value="enabled" <?php checked( $enqueue_genericons, 'enabled' ); ?> />
							</td>
							<td>
								<?php _e("Genericons", "megamenu-pro"); ?>
							</td>
						</tr>
					</table>
				</td>
			</tr>
		</table>

		<?php

	}

	/**
	 * Adds the version number to the header on the general settings page.
	 *
	 * @since 1.0
	 * @param array $versions
	 * @return array
	 */
	public function add_version_to_header( $versions ) {

		$versions['pro'] = array(
			'text' => __("Pro version", "megamenu-pro"),
			'version' => MEGAMENU_PRO_VERSION
		);

		return $versions;

	}


	/**
	 * Enqueue scripts
	 *
	 * @since 1.0
	 */
	public function enqueue_nav_menu_scripts() {

		if ( wp_script_is('megamenu-pro-admin') ) {
			return; // enaure scripts are only loaded once
		}

		wp_enqueue_script( 'megamenu-pro-admin', plugins_url( 'assets/admin.js' , __FILE__ ), array('jquery'), MEGAMENU_PRO_VERSION );

		if ( is_plugin_active( 'megamenu/megamenu.php' ) ) {
			wp_enqueue_script( 'spectrum', MEGAMENU_BASE_URL . 'js/spectrum/spectrum.js', array( 'jquery' ), MEGAMENU_VERSION );
			wp_enqueue_style( 'spectrum', MEGAMENU_BASE_URL . 'js/spectrum/spectrum.css', false, MEGAMENU_VERSION );

			if ( function_exists('wp_enqueue_code_editor') ) {
				wp_deregister_style('codemirror');
				wp_deregister_script('codemirror');

				$cm_settings['codeEditor'] = wp_enqueue_code_editor(array('type' => 'text/html'));
				wp_localize_script('megamenu-pro-admin', 'cm_settings', $cm_settings);
				wp_enqueue_style('wp-codemirror');
			}
		}

		wp_enqueue_style( 'megamenu-genericons', plugins_url( 'icons/genericons/genericons/genericons.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
		wp_enqueue_style( 'megamenu-pro-admin', plugins_url( 'assets/admin.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );

		wp_enqueue_media();


		$params = array(
			'file_frame_title' => __("Media Library", "megamenu-pro")
		);

		wp_localize_script( 'megamenu-pro-admin', 'megamenu_pro', $params );
	}


	/**
	 * Enqueue required CSS and JS for Mega Menu
	 *
	 */
	public function enqueue_admin_scripts( $hook ) {

		wp_enqueue_media();
		wp_enqueue_style( 'megamenu-pro-admin', plugins_url( 'assets/admin.css' , __FILE__ ), false, MEGAMENU_PRO_VERSION );
		wp_enqueue_script( 'megamenu-pro-admin', plugins_url( 'assets/admin.js' , __FILE__ ), array('jquery'), MEGAMENU_PRO_VERSION );

		$params = array(
			'file_frame_title' => __("Media Library", "megamenu-pro")
		);

		wp_localize_script( 'megamenu-pro-admin', 'megamenu_pro', $params );

	}


	/**
	 * Load the plugin classes
	 *
	 * @since 1.0
	 */
	public function load() {

		$plugin_path = plugin_dir_path( __FILE__ );

		$classes = array(
			'Mega_Menu_Updater' => $plugin_path . 'updater/updater.php',
			'Mega_Menu_Sticky' => $plugin_path . 'sticky/sticky.php',
			'Mega_Menu_Google_Fonts' => $plugin_path . 'fonts/google/google-fonts.php',
			'Mega_Menu_Custom_Fonts' => $plugin_path . 'fonts/custom/custom-fonts.php',
			'Mega_Menu_Custom_Icon' => $plugin_path . 'icons/custom/custom.php',
			'Mega_Menu_Genericons' => $plugin_path . 'icons/genericons/genericons.php',
			'Mega_Menu_Style_Overrides' => $plugin_path . 'style-overrides/style-overrides.php',
			'Mega_Menu_Overlay' => $plugin_path . 'overlay/overlay.php',
			'Mega_Menu_Roles' => $plugin_path . 'roles/roles.php',
			'Mega_Menu_Vertical' => $plugin_path . 'vertical/vertical.php',
			'Mega_Menu_Replacements' => $plugin_path . 'replacements/replacements.php',
			'Mega_Menu_Tabbed' => $plugin_path . 'tabbed/tabbed.php',
			'Mega_Menu_Font_Awesome' => $plugin_path . 'icons/fontawesome/fontawesome.php',
			'Mega_Menu_Font_Awesome_5' => $plugin_path . 'icons/fontawesome5/fontawesome5.php',
			'Mega_Menu_Font_Awesome_6' => $plugin_path . 'icons/fontawesome6/fontawesome6.php',
			'Mega_Menu_Badge' => $plugin_path . 'badge/badge.php',
			'Mega_Menu_Image_Swap' => $plugin_path . 'image-swap/image-swap.php'
		);


		if ( ! $this->font_awesome_4_is_enabled() ) {
			unset( $classes['Mega_Menu_Font_Awesome'] );
		}

		if ( ! $this->font_awesome_5_is_enabled() ) {
			unset( $classes['Mega_Menu_Font_Awesome_5'] );
		}

		if ( defined( "MEGAMENU_PRO_IMAGE_SWAP_ENABLED" ) && MEGAMENU_PRO_IMAGE_SWAP_ENABLED === false ) {
			unset( $classes['Mega_Menu_Image_Swap'] );
		}

		foreach ( $classes as $id => $path ) {
			if ( is_readable( $path ) && ! class_exists( $id ) ) {
				include_once $path;
				new $id;
			}
		}

		if ( class_exists( 'Mega_Menu_Toggle_Blocks' ) ) {
			include_once $plugin_path . 'toggle-blocks/toggle-blocks.php';
			new Mega_Menu_Pro_Toggle_Blocks;
		}

	}

	/**
	 * 
	 */
	public static function font_awesome_4_is_enabled() {
		if ( defined( "MEGAMENU_PRO_FA4_ENABLED" ) ) {
			return MEGAMENU_PRO_FA4_ENABLED;
		}

		$initial_version = get_option( 'megamenu_pro_initial_version' );

		if ( $initial_version && version_compare( $initial_version, '2.3', '>=' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * 
	 */
	public static function font_awesome_5_is_enabled() {
		if ( defined( "MEGAMENU_PRO_FA5_ENABLED" ) ) {
			return MEGAMENU_PRO_FA5_ENABLED;
		}

		$initial_version = get_option( 'megamenu_pro_initial_version' );

		if ( $initial_version && version_compare( $initial_version, '2.3', '>=' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Ensure Max Mega Menu (free) is installed
	 *
	 * @since 1.3
	 */
	public function check_megamenu_is_installed() {

		$path = 'megamenu/megamenu.php';

		if ( is_plugin_active($path) ) {
			return;
		}

		$all_plugins = get_plugins();

		if ( isset( $all_plugins[$path] ) ) :

			$plugin = plugin_basename('megamenu/megamenu.php');

			$string = __( 'Max Mega Menu Pro requires Max Mega Menu (free). Please {activate} the Max Mega Menu plugin.', 'megamenu-pro' );

			$link = '<a href="' . wp_nonce_url( 'plugins.php?action=activate&amp;plugin=' . $plugin . '&amp;plugin_status=all&amp;paged=1', 'activate-plugin_' . $plugin ) . '" class="edit">' . __( 'activate', 'megamenu-pro' ) . '</a>';

		?>

		<div class="updated">
			<p>
				<?php echo str_replace( "{activate}", $link, $string ); ?>
			</p>
		</div>

		<?php

		else:

		?>
		<div class="updated">
			<p>
				<?php _e( 'Max Mega Menu Pro requires Max Mega Menu (free). Please install the Max Mega Menu plugin.', 'megamenu-pro' ); ?>
			</p>
			<p class='submit'>
				<a href="<?php echo admin_url( "plugin-install.php?tab=search&type=term&s=max+mega+menu" ) ?>" class='button button-secondary'><?php _e("Install Max Mega Menu", "megamenu-pro"); ?></a>
			</p>
		</div>
		<?php

		endif;

	}


	/**
	 * Apply extra classes to menu items
	 *
	 * @since 1.5
	 * @param array $items
	 * @param array $args
	 * @return array
	 */
	public function apply_classes_to_menu_items( $items, $args ) {

		$parents = array();

		foreach ( $items as $item ) {

			if ( property_exists( $item, 'depth' ) && $item->depth === 0 && $item->megamenu_settings['type'] == 'tabbed') {
				$item->classes[] = 'menu-megamenu';
			}

			if ( isset($item->megamenu_settings['sticky_visibility']) && $item->megamenu_settings['sticky_visibility'] == 'hide') {
				$item->classes[] = 'hide-when-sticky';
			}

			if ( isset($item->megamenu_settings['sticky_visibility']) && $item->megamenu_settings['sticky_visibility'] == 'show') {
				$item->classes[] = 'show-when-sticky';
			}

		}

		return $items;
	}


	/**
	 * Enqueue scripts
	 *
	 * @since 1.3
	 */
	public function enqueue_public_scripts() {
		$scripts_in_footer = apply_filters( "megamenu_scripts_in_footer", true );

		if ( defined( 'MEGAMENU_SCRIPTS_IN_FOOTER' ) ) {
			$scripts_in_footer = MEGAMENU_SCRIPTS_IN_FOOTER;
		}
		
		$handle = apply_filters("megamenu_javascript_handle", "megamenu");
		wp_enqueue_script( 'megamenu-pro', plugins_url( 'assets/public.js' , __FILE__ ), array($handle), MEGAMENU_PRO_VERSION, $scripts_in_footer );
	}


	/**
	 * Register widget
	 *
	 * @since 2.2
	 */
	public function register_widget() {
		if ( defined( "MEGAMENU_PRO_IMAGE_SWAP_ENABLED" ) && MEGAMENU_PRO_IMAGE_SWAP_ENABLED === false ) {
			return;
		}

		$plugin_path = plugin_dir_path( __FILE__ );

		include_once $plugin_path . 'image-swap/image-swap-widget.class.php';
		
		if ( class_exists( 'Mega_Menu_Widget_Image_Swap' ) ) {
			register_widget( 'Mega_Menu_Widget_Image_Swap' );
		}
	}

}

add_action( 'plugins_loaded', array( 'Mega_Menu_Pro', 'init' ), 11 );

endif;